<?php
namespace TotalTheme\Header;

use TotalTheme\Conditional_Logic;

defined( 'ABSPATH' ) || exit;

/**
 * Overlay (Transparent) Header.
 *
 * @package TotalTheme
 * @subpackage Header
 * @version 5.4.2
 */
class Overlay {

	/**
	 * Overlay header is enabled globally or not.
	 */
	protected static $is_global;

	/**
	 * Overlay header is enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * Overlay header logo img.
	 */
	protected static $logo_img;

	/**
	 * Overlay header retina logo img.
	 */
	protected static $logo_img_retina;

	/**
	 * Checks if the overlay header is enabled or not.
	 */
	public static function is_enabled() {
		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		if ( ! Core::is_enabled() ) {
			self::$is_enabled = false;
			return self::$is_enabled; // don't pass through filter.
		}

		// Check if enabled globally.
		$global_check = get_theme_mod( 'overlay_header', false );

		// Check if enabled for specific post types only.
		if ( $global_check ) {
			$condition = get_theme_mod( 'overlay_header_condition', null );
			if ( $condition ) {
				$conditional_logic = new Conditional_Logic( $condition );
				if ( isset( $conditional_logic->result ) ) {
					$global_check = $conditional_logic->result;
				}
			}
		}

		// Default check based on global setting.
		$check = $global_check;

		// Get current post id
		$post_id = wpex_get_current_post_id();

		// Return true if enabled via the post meta.
		// NOTE: The overlay header meta can still be filtered it's not hard set.
		if ( $post_id ) {
			$meta = get_post_meta( $post_id, 'wpex_overlay_header', true );
			if ( $meta ) {
				$check = wpex_validate_boolean( $meta );
			}
		}

		// Prevent issues on password protected pages.
		// @todo may need to revise this...for example now that you can insert a template
		// under the header overlay perhaps this is not needed.
		if ( ! $global_check && post_password_required() && ! wpex_has_page_header() ) {
			$check = false;
		}

		/**
		 * Filters the has_overlay_header check.
		 *
		 * @param bool $check Is the overlary/transparent header enabled.
		 */
		self::$is_enabled = (bool) apply_filters( 'wpex_has_overlay_header', $check );

		return self::$is_enabled;
	}

	/**
	 * Checks if the overlay header is enabled globally.
	 */
	public static function is_global() {
		if ( ! is_null( self::$is_global ) ) {
			return self::$is_global;
		}

		$check = false;

		if ( get_theme_mod( 'overlay_header', false )
			&& ! get_theme_mod( 'overlay_header_condition', null )
			&& ! wpex_has_post_meta( 'wpex_overlay_header' )
		) {
			$check = true;
		}

		/**
		 * Filters whether the overlay header is enabled globally.
		 *
		 * @param bool $check
		 */
		self::$is_global = (bool) apply_filters( 'wpex_is_overlay_header_global', $check );

		return self::$is_global;
	}

	/**
	 * Returns the overlay header style.
	 *
	 * This function should only be called a single time so no need to store the style in a var.
	 */
	public static function style() {
		$style = get_theme_mod( 'overlay_header_style' );

		// Get overlay style based on meta option if hard enabled on the post.
		if ( self::is_enabled() && wpex_has_post_meta( 'wpex_overlay_header' ) ) {
			$meta = get_post_meta( wpex_get_current_post_id(), 'wpex_overlay_header_style', true );
			if ( $meta ) {
				$style = wp_strip_all_tags( trim( $meta ) );
			}
		}

		// White is the default/fallback style.
		if ( ! $style ) {
			$style = 'white';
		}

		/**
		 * Filters the overlay header style.
		 *
		 * @param string $style
		 */
		$style = (string) apply_filters( 'wpex_header_overlay_style', $style );

		return $style;
	}

	/**
	 * Returns the overlay header logo image.
	 */
	public static function logo_img( $parse_logo = true ) {
		if ( ! is_null( self::$logo_img ) ) {
			if ( ! self::$logo_img ) {
				return '';
			}
			if ( $parse_logo ) {
				return self::parse_logo( self::$logo_img );
			} else {
				return self::$logo_img;
			}
		}

		if ( self::is_global() ) {
			$logo = ''; // we use the default logo for a global overlay header.
		} else {
			$logo = wpex_get_translated_theme_mod( 'overlay_header_logo' );
		}

		// Check overlay logo meta option.
		if ( $post_id = wpex_get_current_post_id() ) {
			$meta_logo = get_post_meta( $post_id, 'wpex_overlay_header_logo', true );
			if ( $meta_logo ) {
				// Deprecated redux check.
				if ( is_array( $meta_logo ) ) {
					if ( ! empty( $meta_logo['url'] ) ) {
						$logo = $meta_logo['url'];
					}
				} else {
					$logo = $meta_logo;
				}
			}
		}

		/**
		 * Filters the overlay header logo image url.
		 *
		 * @param string|int $logo
		 */
		self::$logo_img = apply_filters( 'wpex_header_overlay_logo', $logo );

		// return self::logo_img( $parse_logo ); // @todo use recursive function?

		if ( self::$logo_img ) {
			if ( $parse_logo ) {
				return self::parse_logo( self::$logo_img );
			} else {
				return self::$logo_img;
			}
		}
	}

	/**
	 * Returns the overlay header retina logo image.
	 */
	public static function logo_img_retina( $parse_logo = true ) {
		if ( ! is_null( self::$logo_img_retina ) ) {
			if ( ! self::$logo_img_retina ) {
				return '';
			}
			if ( $parse_logo ) {
				return self::parse_logo( self::$logo_img_retina );
			} else {
				return self::$logo_img_retina;
			}
		}

		if ( self::is_global() ) {
			$logo = ''; // we use the default logo for a global overlay header.
		} else {
			$logo = wpex_get_translated_theme_mod( 'overlay_header_logo_retina' );
		}

		if ( $post_id = wpex_get_current_post_id() ) {
			$meta_logo = get_post_meta( $post_id, 'wpex_overlay_header_logo_retina', true );
			if ( $meta_logo ) {
				$logo = $meta_logo;
			}
		}

		/**
		 * Filters the overlay header logo retina image.
		 *
		 * @param int|string $logo
		 */
		self::$logo_img_retina = apply_filters( 'wpex_header_overlay_logo_retina', $logo );

		// return self::logo_img_retina();

		if ( self::$logo_img_retina ) {
			if ( $parse_logo ) {
				return self::parse_logo( self::$logo_img_retina );
			} else {
				return self::$logo_img_retina;
			}
		}
	}

	/**
	 * Parses the logo img.
	 */
	private static function parse_logo( $logo = '' ) {
		return wpex_get_image_url( $logo );
	}

}