<?php
namespace TotalTheme\Mobile;

defined( 'ABSPATH' ) || exit;

/**
 * Mobile Menu.
 *
 * @package TotalTheme
 * @subpackage Mobile
 * @version 5.4
 */
class Menu {

	/**
	 * Mobile menu is enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * Mobile menu style.
	 */
	protected static $style;

	/**
	 * The mobile menu breakpoint.
	 */
	protected static $breakpoint;

	/**
	 * Hook mobile menu template part into the wp_footer hook.
	 */
	public static function get_template_part() {
		if ( ! self::is_enabled() ) {
			return;
		}
		$style = self::style();
		switch( $style ) {
			case 'toggle':
				$template_part = 'mobile_menu_toggle';
				break;
			case 'full_screen':
				$template_part = 'mobile_menu_full_screen';
				break;
			case 'sidr':
				$template_part = 'mobile_menu_sidr';
				break;
			default:
				$template_part = '';
				break;
		}
		if ( $template_part ) {
			wpex_get_template_part( $template_part );
		}
	}

	/**
	 * Returns the mobile menu style.
	 */
	public static function style() {
		if ( ! is_null( self::$style ) ) {
			return self::$style;
		}

		$style = wpex_is_layout_responsive() ? get_theme_mod( 'mobile_menu_style' ) : 'disabled';

		if ( empty( $style ) ) {
			$style = 'sidr';
		}

		if ( wpex_is_header_menu_custom() ) {
			$style = 'disabled';
		}

		$style = apply_filters( 'wpex_mobile_menu_style', $style ); // @todo deprecate.

		/**
		 * Filters the header mobile menu style.
		 *
		 * @param string $style
		 */
		self::$style = (string) apply_filters( 'wpex_header_menu_mobile_style', $style );

		return self::$style;
	}

	/**
	 * Check if the mobile menu is enabled.
	 */
	public static function is_enabled() {
		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		if ( wpex_has_custom_header() ) {
			return false; // always false for custom header.
		}

		$check = false;

		if ( ( wpex_has_mobile_menu_alt() || wpex_has_header_menu() )
			&& wpex_is_layout_responsive()
			&& 'disabled' !== self::style()
		) {
			$check = true;
		}

		$check = apply_filters( 'wpex_has_mobile_menu', $check ); // @todo deprecate

		/**
		 * Filters whether the site header has a mobile menu or not.
		 *
		 * @param bool $check
		 */
		self::$is_enabled = (bool) apply_filters( 'wpex_has_header_menu_mobile', $check );

		return self::$is_enabled;
	}

	/**
	 * Returns the mobile menu breakpoint.
	 */
	public static function breakpoint() {
		if ( ! is_null( self::$breakpoint ) ) {
			return self::$breakpoint;
		}

		$breakpoint = get_theme_mod( 'mobile_menu_breakpoint' );

		if ( $breakpoint ) {
			$breakpoint = absint( $breakpoint );
		}

		/**
		 * Filters the mobile menu breakpoint.
		 *
		 * @param int $breakpoint
		 */
		$breakpoint = (int) apply_filters( 'wpex_header_menu_mobile_breakpoint', $breakpoint );

		self::$breakpoint = absint( $breakpoint ) ?: 959; // can't be empty since Total 5.0

		return self::$breakpoint;
	}

	/**
	 * Outputs mobile menu top hook content.
	 */
	public static function hook_top() {
		ob_start();
			$top = wpex_hook_mobile_menu_top();
		$top = ob_get_clean();

		if ( $top ) {
			$style = self::style();
			$utl_class = '';
			switch ( $style ) {
				case 'full_screen':
				case 'toggle':
					$utl_class = ' wpex-pb-20';
					break;
			}
			echo '<div class="wpex-mobile-menu-top' . $utl_class . '">' . $top . '</div>';
		}
	}

	/**
	 * Outputs mobile menu bottom hook content.
	 */
	public static function hook_bottom() {
		ob_start();
			$bottom = wpex_hook_mobile_menu_bottom();
		$bottom = ob_get_clean();

		if ( $bottom ) {
			$style = self::style();
			$utl_class = '';
			switch ( $style ) {
				case 'full_screen':
					$utl_class = ' wpex-pt-20';
					break;
				case 'toggle':
					$utl_class = ' wpex-pb-20';
					break;
			}
			echo '<div class="wpex-mobile-menu-bottom' . $utl_class . '">' . $bottom . '</div>';
		}
	}

	/**
	 * Outputs mobile menu searchform.
	 */
	public static function search_form( $args = array() ) {
		if ( ! get_theme_mod( 'mobile_menu_search', true ) ) {
			return;
		}
		wpex_get_template_part( 'mobile_searchform', null, $args );
	}

}