<?php
namespace TotalTheme\Topbar;

use TotalTheme\Topbar\Core as Topbar;

defined( 'ABSPATH' ) || exit;

/**
 * Topbar Social.
 *
 * @package TotalTheme
 * @subpackage Topbar
 * @version 5.4.5
 */
class Social {

	/**
	 * Topbar social is enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * Topbar social icon style.
	 */
	protected static $icon_style;

	/**
	 * Stores the topbar social template id if defined.
	 */
	protected static $template_id;

	/**
	 * Checks if the overlay topbar is enabled or not.
	 */
	public static function is_enabled() {
		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		$is_enabled = wp_validate_boolean( get_theme_mod( 'top_bar_social', true ) );

		if ( $is_enabled && ( empty( self::get_profile_options() ) || empty( self::get_registered_profiles() ) ) ) {
			$is_enabled = false;
		}

		if ( ! $is_enabled && self::get_alt_content() ) {
			$is_enabled = true;
		}

		/**
		 * Filters whether the topbar social is enabled.
		 *
		 * @param bool $is_enabled
		 */
		self::$is_enabled = (bool) apply_filters( 'wpex_has_topbar_social', $is_enabled );

		return self::$is_enabled;
	}

	/**
	 * Returns an array of social options.
	 */
	public static function get_profile_options() {
		$options = wpex_social_profile_options_list();

		/**
		 * Filters the options for the social profiles customizer options.
		 *
		 * @param array $options
		 */
		$options = (array) apply_filters ( 'wpex_topbar_social_options', $options );

		return $options;
	}

	/**
	 * Returns an array of registered social profiles for the topbar.
	 */
	public static function get_registered_profiles() {
		$profiles = get_theme_mod( 'top_bar_social_profiles' );
		if ( $profiles && is_string( $profiles ) ) {
			$profiles = json_decode( $profiles, true );
		}
		if ( $profiles && is_array( $profiles ) ) {
			$profiles = array_filter( $profiles );
		}
		return $profiles;
	}

	/**
	 * Returns topbar social icon style.
	 */
	public static function get_icon_style() {
		if ( ! is_null( self::$icon_style ) ) {
			return self::$icon_style;
		}

		$style = get_theme_mod( 'top_bar_social_style' ) ?: 'none';

		if ( 'colored-icons' == $style  || 'images' === $style ) {
			$style = 'flat-color-rounded'; // old styles deprecated in v4.9
		}

		/**
		 * Filters the topbar social style.
		 *
		 * @param string $style.
		 */
		self::$icon_style = (string) apply_filters( 'wpex_topbar_social_style', $style );

		return self::$icon_style;
	}

	/**
	 * Return the topbar social alt content from the theme mod.
	 */
	private static function get_alt_content_from_mod() {
		return trim( wpex_get_translated_theme_mod( 'top_bar_social_alt' ) );
	}

	/**
	 * Return topbar social template ID.
	 */
	public static function get_template_id() {
		if ( ! is_null( self::$template_id ) ) {
			return self::$template_id;
		}
		$template_id = ''; // make sure it isn't null to prevent extra checks.
		$content = self::get_alt_content_from_mod();
		if ( is_numeric( $content ) ) {
			$post_id = wpex_parse_obj_id( $content, 'page' );
			$post    = get_post( $post_id );
			if ( $post && ! is_wp_error( $post ) ) {
				$template_id = $post_id;
			}
		}
		self::$template_id = $template_id;
		return self::$template_id;
	}

	/**
	 * Get alternative content.
	 */
	public static function get_alt_content() {
		$template_id = self::get_template_id();

		if ( $template_id ) {
			$content = wpex_parse_vc_content( get_post_field( 'post_content', $template_id ) );
		} else {
			$content = self::get_alt_content_from_mod();
		}

		/**
		 * Filters the topbar social alt content.
		 *
		 * @param string $content.
		 */
		$content = (string) apply_filters( 'wpex_topbar_social_alt_content', $content );

		return $content;
	}

	/**
	 * Echo class attribute for the the topbar social wrapper element.
	 */
	public static function wrapper_class() {
		$classes      = array();
		$topbar_style = Topbar::style();
		$split_bk     = Topbar::breakpoint();
		$has_split_bk = ( 'none' !== $split_bk );

		switch ( $topbar_style ) {
			case 'one':
				$classes[] = 'top-bar-right';
				if ( Topbar::get_content() && $has_split_bk ) {
					$classes[] = 'wpex-mt-10';
					$classes[] = 'wpex-' . $split_bk . '-mt-0';
				}
				break;
			case 'two':
				$classes[] = 'top-bar-left';
				break;
			case 'three':
				$classes[] = 'top-bar-centered';
				if ( has_nav_menu( 'topbar_menu' ) || Topbar::get_content() ) {
					$classes[] = 'wpex-mt-10'; // extra spacing for centered top bar when there is content
				}
				break;
		}

		if ( empty( self::get_alt_content() ) ) {
			$social_style = self::get_icon_style();
			if ( $social_style ) {
				$classes[] = 'social-style-' . sanitize_html_class( $social_style ); // @todo rename to top-bar-social--{style}
			}
		}

		if ( ! $has_split_bk ) {
			$classes[] = 'wpex-flex-shrink-0';
		}

		/**
		 * Filters the topbar social element class.
		 *
		 * @param array $class
		 */
		$classes = (array) apply_filters( 'wpex_topbar_social_class', $classes );

		if ( $classes ) {
			echo 'class="' . esc_attr( implode( ' ', $classes ) ) . '"';
		}
	}

	/**
	 * Render list.
	 */
	public static function render_list() {
		$profiles       = self::get_registered_profiles();
		$social_options = self::get_profile_options();

		if ( empty( $profiles ) || empty( $social_options ) ) {
			return;
		}

		$social_style = self::get_icon_style();
		$link_target  = get_theme_mod( 'top_bar_social_target', 'blank' );
		$gap          = get_theme_mod( 'top_bar_social_gap' );

		if ( ! $gap ) {
			if ( 'none' === $social_style || 'default' === $social_style || empty( $social_style ) ) {
				$gap = 10;
			} else {
				$gap = 5;
			}
		}

		// Begin output.
		$list = '<ul id="top-bar-social-list" class="wpex-inline-block wpex-list-none wpex-align-bottom wpex-m-0 wpex-last-mr-0">';

		// Loop through social options.
		$list_items = '';
		foreach ( $profiles as $site => $url ) {

			if ( ! $url || ! array_key_exists( $site, $social_options ) ) {
				continue;
			}

			$site = sanitize_key( $site );

			// Sanitize email and remove link target.
			if ( 'email' === $site ) {
				$sanitize_email = sanitize_email( $url );
				if ( is_email( $url ) ) {
					$link_target = '';
					$sanitize_email = antispambot( $sanitize_email );
					$url = 'mailto:' . $sanitize_email;
				} elseif( strpos( $url, 'mailto' ) !== false ) {
					$link_target = '';
				}
			}

			// Sanitize phone number.
			if ( 'phone' === $site
				&& false === strpos( $url, 'tel:' )
				&& false === strpos( $url, 'callto:' )
			) {
				$url = 'tel:' . $url;
			}

			// Li classes.
			$li_class = 'wpex-inline-block';

			if ( $gap_escaped = absint( $gap ) ) {
				$li_class .= ' wpex-mr-' . absint( $gap );
			}

			// Generate link HTML based on attributes and content.
			$list_items .= '<li class="' . esc_attr( $li_class ) . '">';

				$label = $social_options[$site]['label'] ?? '';

				$link_attrs = array(
					'href'   => $url,
					'title'  => $label,
					'target' => $link_target,
					'class'  => 'wpex-' . $site . ' ' . wpex_get_social_button_class( $social_style ),
				);

				/**
				 * Filters the top bar social link attributes.
				 *
				 * @param array $link_attributes
				 * @param string $site
				 */
				$link_attrs = (array) apply_filters( 'wpex_topbar_social_link_attrs', $link_attrs, $site );

				$icon_class = $social_options[$site]['icon_class'] ?? '';

				if ( $icon_class ) {
					$icon_html = wpex_get_theme_icon_html( $icon_class );
					$icon_html .= '<span class="screen-reader-text">' . esc_attr( $label ) . '</span>';
				} else {
					$icon_html = $label;
				}

			 	$list_items .= wpex_parse_html( 'a', $link_attrs, $icon_html );

			 $list_items .= '</li>';

		} // endforeach.

			/**
			 * Filters the top bar social links output.
			 *
			 * @param string $links_output
			 */
			$list_items = (string) apply_filters( 'wpex_topbar_social_links_output', $list_items );

			$list .= $list_items;

		$list .= '</ul>';

		/**
		 * Filters the top bar social list output.
		 *
		 * @param string $list
		 */
		echo (string) apply_filters( 'wpex_topbar_social_list', $list );
	}

}