<?php
namespace TotalTheme;

use TotalTheme\Header\Core as Header;
use TotalTheme\Header\Logo as Header_Logo;
use TotalTheme\Header\Overlay as Overlay_Header;
use TotalTheme\Mobile\Menu as Mobile_Menu;

defined( 'ABSPATH' ) || exit;

/**
 * Advanced frontend styles based on user settings.
 *
 * @package TotalTheme
 * @version 5.4.3
 */
final class Advanced_Styles {

	/**
	 * Class instance.
	 *
	 * @access private
	 * @var object Class object.
	 */
	private static $instance;

	/**
	 * Create or retrieve the class instance.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}
		return static::$instance;
	}

	/**
	 * CSS var.
	 *
	 * @param string $css
	 */
	protected static $css;

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'wpex_head_css', __CLASS__ . '::update_head_css', 999 );
	}

	/**
	 * Hooks into wpex_head_css to add custom css to the <head> tag.
	 */
	public static function update_head_css( $head_css ) {
		self::generate_css();

		if ( ! empty( self::$css ) ) {
			$head_css .= '/*ADVANCED STYLING CSS*/' . self::$css;
		}

		return $head_css;
	}

	/**
	 * Generates css.
	 */
	protected static function generate_css() {
		$methods = array(
			'header_background',
			'overlay_header',
			'logo_mobile_side_margin',
			'logo_height',
			'page_header_title',
			'footer_background_image',
			'footer_callout_background_image',
			'blockquote_border',
		);

		if ( WPEX_VC_ACTIVE ) {
			$methods[] = 'vc_inline_shrink_sticky_header_height';
			if ( get_option( 'wpb_js_use_custom' ) ) {
				$methods[] = 'vc_column_gutter';
			}
		}

		foreach ( $methods as $method ) {
			$method_css = self::$method();
			if ( $method_css ) {
				self::$css .= $method_css;
			}
		}

	}

	/**
	 * Header background.
	 */
	public static function header_background() {
		$header_bg = Header::get_background_image_url();
		if ( $header_bg ) {
			return '#site-header{background-image:url(' . esc_url( $header_bg ) . ');}';
		}
	}

	/**
	 * Overlay header.
	 */
	public static function overlay_header() {
		$css     = '';
		$post_id = wpex_get_current_post_id();

		if ( $post_id && wpex_has_post_meta( 'wpex_overlay_header' ) && Overlay_Header::is_enabled() ) {

			// Custom overlay header font size.
			$overlay_header_font_size = get_post_meta( $post_id, 'wpex_overlay_header_font_size', true );

			if ( $overlay_header_font_size ) {
				$css .= '#site-navigation, #site-navigation .dropdown-menu a{font-size:' . intval( $overlay_header_font_size ) . 'px;}';
			}

			// Apply overlay header background color.
			// Note we use background and not background-color.
			$overlay_header_bg = get_post_meta( $post_id, 'wpex_overlay_header_background', true );

			if ( $overlay_header_bg ) {
				$css .= '#site-header.overlay-header.dyn-styles{background:' . esc_attr( $overlay_header_bg ) . '; }';
			}

		}

		return $css;
	}

	/**
	 * Logo mobile side margin.
	 */
	public static function logo_mobile_side_margin() {
		$css          = '';
		$header_style = Header::Style();

		if ( 'seven' === $header_style
			|| 'eight' === $header_style
			|| 'nine' === $header_style
			|| 'ten' === $header_style
		) {
			return;
		}

		$margin_escaped = absint( get_theme_mod( 'logo_mobile_side_offset' ) );

		if ( ! empty( $margin_escaped ) ) {

			$mm_breakpoint = Mobile_Menu::breakpoint();

			if ( $mm_breakpoint < 9999 ) {
				$css .= '@media only screen and (max-width:' . $mm_breakpoint . 'px) {';
			}

			$css .= 'body.has-mobile-menu #site-logo {';
				if ( is_rtl() ) {
					$css .= 'margin-left:' . $margin_escaped . 'px;';
				} else {
					$css .= 'margin-right:' . $margin_escaped . 'px;';
				}
			$css .= '}';

			if ( $mm_breakpoint < 9999 ) {
				$css .= '}';
			}

		}

		return $css;
	}

	/**
	 * Custom logo height.
	 */
	public static function logo_height() {
		$logo_is_svg = Header_Logo::is_image_svg();
		if ( $logo_is_svg || get_theme_mod( 'apply_logo_height', false ) ) {
			$height = absint( get_theme_mod( 'logo_height' ) );
			$height_prop = 'max-height';
			if ( Header::has_fixed_height() || $logo_is_svg ) {
				$height_prop = 'height';
			}
			if ( $height ) {
				return '#site-logo .logo-img{' . $height_prop . ':' . esc_attr( $height ) . 'px;width:auto;}'; // auto width needed for responsiveness.
			}
		}
	}

	/**
	 * Shrink header height.
	 *
	 * This is used to provide consistency with the shrink header logo when using the front-end builder
	 * since the sticky header is disabled when using the builder.
	 *
	 * @todo perhaps we can target a different classname in the default CSS that gets added on the front-end only?
	 */
	public static function vc_inline_shrink_sticky_header_height() {
		if ( ! wpex_vc_is_inline() || Header::has_flex_container() ) { // important not needed for flex header!!
			return;
		}

		$shrink_header_style = wpex_sticky_header_style();

		if ( 'shrink' === $shrink_header_style || 'shrink_animated' === $shrink_header_style ) {
			return '#site-logo .logo-img{max-height:var(--wpex-site-header-shrink-start-height, 60px)!important;width:auto;}';
		}
	}

	/**
	 * Page header title.
	 */
	public static function page_header_title() {
		if ( ! wpex_has_post_meta( 'wpex_post_title_style' ) ) {
			$page_header_bg = wpex_page_header_background_image(); // already passed through wpex_get_image_url
			if ( $page_header_bg ) {
				return '.page-header.has-bg-image{background-image:url(' . esc_url( $page_header_bg ) . ');}';
			}
		}
	}

	/**
	 * Footer background.
	 */
	public static function footer_background_image() {
		$background = wpex_get_image_url( get_theme_mod( 'footer_bg_img' ) );
		if ( $background ) {
			return '#footer{background-image:url(' . esc_url( $background ) . ');}';
		}
	}

	/**
	 * Footer callout background.
	 */
	public static function footer_callout_background_image() {
		$background = wpex_get_image_url( get_theme_mod( 'footer_callout_bg_img' ) );
		if ( $background ) {
			return '#footer-callout-wrap{background-image:url(' . esc_url( $background ) . ');}';
		}
	}

	/**
	 * Define Visual Composer gutter.
	 */
	public static function vc_column_gutter() {
		$custom_gutter = get_option( 'wpb_js_gutter' );
		if ( $custom_gutter ) {
			return ':root{--wpex-vc-gutter:' . absint( $custom_gutter ) . 'px}';
		}
	}

	/**
	 * Adds border to the blockquote element.
	 */
	public static function blockquote_border() {
		$border_width = get_theme_mod( 'blockquote_border_width' );
		if ( $border_width ) {
			return 'blockquote{background:none;border-width: 0 0 0 ' . absint( $border_width ) . 'px;border-color:var(--wpex-accent);border-style:solid;padding-left:25px;}';
		}
	}

}