<?php
/**
 * Return utility font sizes.
 *
 * @since 5.0
 */
function wpex_utl_font_sizes() {
	$font_sizes = array(
		''     => esc_html__( 'Default', 'total' ),
		'base' => esc_html__( 'Base (1em)', 'total' ),
		'xs'   => esc_html__( 'Extra Small', 'total' ),
		'sm'   => esc_html__( 'Small', 'total' ),
		'md'   => esc_html__( 'Medium', 'total' ),
		'lg'   => esc_html__( 'Large', 'total' ),
		'xl'   => esc_html__( 'x Large', 'total' ),
		'2xl'  => esc_html__( '2x Large', 'total' ),
		'3xl'  => esc_html__( '3x Large', 'total' ),
		'4xl'  => esc_html__( '4x Large', 'total' ),
		'5xl'  => esc_html__( '5x Large', 'total' ),
		'6xl'  => esc_html__( '6x Large', 'total' ),
		'7xl'  => esc_html__( '7x Large', 'total' ),
	);

	/**
	 * Filters the theme font size options.
	 *
	 * @param array $font_sizes
	 */
	$font_sizes = (array) apply_filters( 'wpex_utl_font_sizes', $font_sizes );

	return $font_sizes;
}

/**
 * Return utility percentage widths.
 *
 * @since 5.0
 */
function wpex_utl_percent_widths() {
	$widths = array(
		''    => esc_html__( 'Default', 'total' ),
		'20'  => '20%',
		'25'  => '25%',
		'30'  => '30%',
		'33'  => '33%',
		'40'  => '40%',
		'50'  => '50%',
		'60'  => '60%',
		'67'  => '67%',
		'70'  => '70%',
		'75'  => '75%',
		'80'  => '80%',
		'100' => '100%',
	);

	/**
	 * Filters the percent width options.
	 *
	 * @param array $widths
	 */
	$widths = (array) apply_filters( 'wpex_utl_precentage_widths', $widths );

	return $widths;
}

/**
 * Return utility border radius.
 *
 * @since 5.0
 */
function wpex_utl_border_radius( $include_blobs = false ) {
	$choices = array(
		''              => esc_html__( 'Default', 'total' ),
		'rounded-xs'    => esc_html__( 'Extra Small', 'total' ),
		'rounded-sm'    => esc_html__( 'Small', 'total' ),
		'rounded'       => esc_html__( 'Average', 'total' ),
		'rounded-md'    => esc_html__( 'Medium', 'total' ),
		'rounded-lg'    => esc_html__( 'Large', 'total' ),
		'rounded-full'  => esc_html__( 'Full', 'total' ),
		'rounded-0'     => esc_html__( 'None', 'total' ),
	);

	if ( $include_blobs ) {
		$blobs = array(
			'radius-blob-1' => esc_html__( 'Blob 1', 'total' ),
			'radius-blob-2' => esc_html__( 'Blob 2', 'total' ),
			'radius-blob-3' => esc_html__( 'Blob 3', 'total' ),
			'radius-blob-4' => esc_html__( 'Blob 4', 'total' ),
			'radius-blob-5' => esc_html__( 'Blob 5', 'total' ),
		);
		$choices = array_merge( $choices, $blobs );
	}

	/**
	 * Filters the border radius choices.
	 *
	 * @param array $choices
	 * @param bool $include_blobs
	 */
	$choices = (array) apply_filters( 'wpex_utl_border_radius', $choices, $include_blobs );

	return $choices;
}

/**
 * Return utility border width types.
 *
 * @since 5.0
 */
function wpex_utl_border_widths() {
	$widths = array(
		''  => esc_html__( 'Default', 'total' ),
		'0px'  => '0px',
		'1'    => '1px',
		'2'    => '2px',
		'3'    => '3px',
		'4'    => '4px',
	);

	/**
	 * Filters the border width options.
	 *
	 * @param array $widths
	 */
	$widths = (array) apply_filters( 'wpex_utl_border_widths', $widths );

	return $widths;
}

/**
 * Return utility paddings.
 *
 * @since 5.0
 */
function wpex_utl_paddings() {
	$paddings = array(
		''     => esc_html__( 'Default', 'total' ),
		'0px'  => '0px',
		'5px'  => '5px',
		'10px' => '10px',
		'15px' => '15px',
		'20px' => '20px',
		'25px' => '25px',
		'30px' => '30px',
		'40px' => '40px',
		'50px' => '50px',
	);

	/**
	 * Filters the padding options.
	 *
	 * @param array $paddings
	 */
	$paddings = (array) apply_filters( 'wpex_utl_paddings', $paddings );

	return $paddings;
}

/**
 * Return utility margins.
 *
 * @since 5.0
 */
function wpex_utl_margins() {
	$margins = array(
		''     => esc_html__( 'Default', 'total' ),
		'0px'  => '0px',
		'5px'  => '5px',
		'10px' => '10px',
		'15px' => '15px',
		'20px' => '20px',
		'25px' => '25px',
		'30px' => '30px',
		'40px' => '40px',
		'50px' => '50px',
	);

	/**
	 * Filters the margin options.
	 *
	 * @param array $margins
	 */
	$margins = (array) apply_filters( 'wpex_utl_margins', $margins );

	return $margins;
}


/**
 * Return utility letter spacing options.
 *
 * @since 5.4
 */
function wpex_utl_letter_spacing() {
	$letter_spacing = array(
		''         => esc_html__( 'Default', 'total' ),
		'tighter'  => esc_html__( 'Tighter', 'total' ),
		'tight'    => esc_html__( 'Tight', 'total' ),
		'normal'   => esc_html__( 'Normal', 'total' ),
		'wide'     => esc_html__( 'Wide', 'total' ),
		'wider'    => esc_html__( 'Wider', 'total' ),
		'widest'   => esc_html__( 'Widest', 'total' ),
	);

	/**
	 * Filters the line_height options.
	 *
	 * @param array $letter_spacing
	 */
	$letter_spacing = (array) apply_filters( 'wpex_utl_letter_spacing', $letter_spacing );

	return $letter_spacing;
}

/**
 * Return utility line height options.
 *
 * @since 5.4
 */
function wpex_utl_line_height() {
	$line_height = array(
		''        => esc_html__( 'Default', 'total' ),
		'tight'   => esc_html__( 'Tight', 'total' ),
		'snug'    => esc_html__( 'Snug', 'total' ),
		'normal'  => esc_html__( 'Normal', 'total' ),
		'relaxed' => esc_html__( 'Relaxed', 'total' ),
		'loose'   => esc_html__( 'Loose', 'total' ),
	);

	/**
	 * Filters the line height options.
	 *
	 * @param array $line_height
	 */
	$line_height = (array) apply_filters( 'wpex_utl_line_height', $line_height );

	return $line_height;
}

/**
 * Get utility shadows.
 *
 * @since 5.0
 */
function wpex_utl_shadows() {
	$shadows = array(
		''		      => esc_html__( 'Default', 'total' ),
		'shadow-none' => esc_html__( 'None', 'total' ),
		'shadow-xs'   => esc_html__( 'Extra Small', 'total' ),
		'shadow-sm'   => esc_html__( 'Small', 'total' ),
		'shadow'      => esc_html__( 'Average', 'total' ),
		'shadow-md'   => esc_html__( 'Medium', 'total' ),
		'shadow-lg'   => esc_html__( 'Large', 'total' ),
		'shadow-xl'   => esc_html__( 'Extra Large', 'total' ),
		'shadow-2xl'  => esc_html__( '2x Large', 'total' ),
	);

	/**
	 * Filters the shadow options.
	 *
	 * @param array $shadows
	 */
	$shadows = (array) apply_filters( 'wpex_utl_shadows', $shadows );

	return $shadows;
}

/**
 * Get utility divider styles.
 *
 * @since 5.0
 */
function wpex_utl_divider_styles() {
	$styles = array(
		''       => esc_html__( 'Default', 'total' ),
		'solid'  => esc_html__( 'Solid', 'total' ),
		'dotted' => esc_html__( 'Dotted', 'total' ),
		'dashed' => esc_html__( 'Dashed', 'total' ),
	);

	/**
	 * Filters the divider styles options.
	 *
	 * @param array $styles
	 */
	$styles = (array) apply_filters( 'wpex_utl_divider_styles', $styles );

	return $styles;
}

/**
 * Get utility opacity.
 *
 * @since 5.0
 */
function wpex_utl_opacities() {
	$opacities = array(
		''	  => esc_html__( 'Default', 'total' ),
		'10'  => '10%',
		'20'  => '20%',
		'30'  => '30%',
		'40'  => '40%',
		'50'  => '50%',
		'60'  => '60%',
		'70'  => '70%',
		'80'  => '80%',
		'90'  => '90%',
		'100' => '100%',
	);

	/**
	 * Filters the opacity options.
	 *
	 * @param array $opacities
	 */
	$opacities = (array) apply_filters( 'wpex_utl_opacities', $opacities );

	return $opacities;
}

/**
 * Get utility breakpoints.
 *
 * @since 5.0
 */
function wpex_utl_breakpoints() {
	$breakpoints = array(
		''   => esc_html__( 'Default', 'total' ),
		'sm' => esc_html__( 'sm - 640px', 'total' ),
		'md' => esc_html__( 'md - 768px', 'total' ),
		'lg' => esc_html__( 'lg - 1024px', 'total' ),
		'xl' => esc_html__( 'xl - 1280px', 'total' ),
	);

	/**
	 * Filters the breakpoint options.
	 *
	 * @param array $breakpoints
	 */
	$breakpoints = (array) apply_filters( 'wpex_utl_breakpoints', $breakpoints );

	return $breakpoints;
}

/**
 * Return utility visibility class.
 *
 * @since 5.0
 */
function wpex_utl_visibility_class( $show_hide = 'hide', $screen = '' ) {
	if ( ! $screen || ! array_key_exists( $screen, wpex_utl_breakpoints() ) ) {
		return;
	}

	$class = '';

	switch ( $show_hide ) {
		case 'hide':
			$class = 'wpex-hidden wpex-' . sanitize_html_class( $screen ) . '-block';
			break;
		case 'show':
			$class = 'wpex-' . sanitize_html_class( $screen ) . '-hidden';
			break;
	}

	return $class;
}

/**
 * Return utility font size class.
 *
 * @since 5.0
 */
function wpex_sanitize_utl_font_size( $font_size = '' ) {
	if ( ! $font_size || ! array_key_exists( $font_size, wpex_utl_font_sizes() )  ) {
		return;
	}

	switch ( $font_size ) {
		case '5xl':
		case '6xl':
		case '7xl':
			$class = 'wpex-text-4xl wpex-md-text-' . sanitize_html_class( $font_size );
			break;
		default:
			$class = 'wpex-text-' . sanitize_html_class( $font_size );
			break;
	}

	/**
	 * Returns a font size classname.
	 *
	 * @param string|null $class
	 * @param string $font_size
	 */
	$class = apply_filters( 'wpex_utl_font_size_class', $class, $font_size );

	return $class;
}