<?php
defined( 'ABSPATH' ) || exit;

/**
 * Post Cards Shortcode.
 *
 * @package TotalThemeCore
 * @version 1.4.5
 */
if ( ! class_exists( 'WPEX_Post_Cards_Shortcode' ) ) {

	class WPEX_Post_Cards_Shortcode {

		/**
		 * Associative array of current shortcode attributes.
		 */
		public $atts = array();

		/**
		 * Current query.
		 */
		public $query = null;

		/**
		 * Check if we are currently loading more posts via ajax.
		 */
		public $doing_loadmore = false;

		/**
		 * Main constructor.
		 */
		public function __construct() {
			add_shortcode( 'wpex_post_cards', array( $this, 'output' ) );

			if ( function_exists( 'vc_lean_map' ) ) {
				TotalThemeCore\WPBakery\Map\WPEX_Post_Cards::instance();
			}
		}

		/**
		 * Shortcode output => Get template file and display shortcode.
		 */
		public function output( $atts, $content = null ) {
			if ( ! vcex_maybe_display_shortcode( 'wpex_post_cards', $atts )
				|| ! function_exists( 'vcex_build_wp_query' )
				|| ! function_exists( 'wpex_get_card' )
			) {
				return;
			}

			// Int some class vars.
			$this->doing_loadmore = vcex_doing_loadmore();

			// Define shortcode output.
			$output = $inner_output = '';

			// Store some original atts before parsing.
			$og_paged = ! empty( $atts['paged'] ) ? $atts['paged'] : null;
			$og_entry_count = ! empty( $atts['entry_count'] ) ? absint( $atts['entry_count'] ) : null;
			$running_count = ! empty( $atts['running_count'] ) ? absint( $atts['running_count'] ) : 0;

			// Parse shortcode atts.
			$this->atts = vcex_shortcode_atts( 'wpex_post_cards', $atts, get_class() );

			// Core vars.
			$is_auto_query = vcex_validate_att_boolean( 'auto_query', $this->atts );
			$display_type = $this->get_display_type();
			$grid_style = $this->get_grid_style();
			$grid_columns = $this->get_grid_columns();
			$grid_gap_class = $this->get_grid_gap_class();
			$grid_is_responsive = vcex_validate_att_boolean( 'grid_columns_responsive', $this->atts, true );

			// Featured card vars.
			$has_featured_card = $this->has_featured_card();
			$featured_card_location = $this->get_featured_card_location();
			$is_featured_card_top = $this->is_featured_card_top();
			$fc_bk = ! empty( $this->atts['featured_breakpoint'] ) ? $this->atts['featured_breakpoint'] : 'sm';

			// Ajax attributes (don't need to be parsed).
			if ( $this->has_loadmore() ) {
				$ajax_atts = $atts;
				if ( $is_auto_query ) {
					$ajax_atts['query_vars'] = $ajax_atts['query_vars'] ?? array();
				}
				$this->atts['paged'] = $og_paged;
			}

			// We can remove $atts from memory now.
			unset( $atts );

			// Parse featured card ID.
			if ( vcex_validate_att_boolean( 'featured_card', $this->atts ) ) {
				$this->atts['featured_post_id'] = $this->get_featured_post_id();
				if ( isset( $ajax_atts ) ) {
					$ajax_atts['featured_post_id'] = $this->atts['featured_post_id'];
				}
			}

			// Set current entry count.
			$entry_count = $og_entry_count ?? 0;

			// Define card args.
			$card_args = $this->get_card_args();

			// Query posts.
			$this->query = vcex_build_wp_query( $this->atts, 'wpex_post_cards' );

			// Check if we have posts or a featured post.
			if ( $this->query->have_posts() || ! empty( $this->atts['featured_post_id'] ) ) :

				/*-------------------------------------*/
				/* [ Inner Output Starts Here ]
				/*-------------------------------------*/

				// Add inner flex wrap
				if ( $has_featured_card && ! $is_featured_card_top ) {

					$inner_class = array(
						'wpex-post-cards-inner',
					);

					if ( ! $is_featured_card_top ) {
						$inner_class[] = 'wpex-' . $fc_bk . '-flex';
					}

					if ( 'right' === $featured_card_location ) {
						$inner_class[] = 'wpex-' . $fc_bk . '-flex-row-reverse';
					}

					$inner_output .= '<div class="' . esc_attr( implode( ' ', $inner_class ) ) . '">';
				}

				/*-------------------------------------*/
				/* [ Featured Card ]
				/*-------------------------------------*/
				if ( $has_featured_card ) {

					$fc_width = apply_filters( 'wpex_post_cards_featured_width', 50 );
					$fc_width = ! empty( $this->atts['featured_width'] ) ? $this->atts['featured_width'] : $fc_width;

					$featured_card_classes = array(
						'wpex-post-cards-featured',
					);

					// Featured card flex classes
					if ( ! $is_featured_card_top ) {
						$featured_card_classes[] = 'wpex-' . $fc_bk . '-w-' . trim( absint( $fc_width ) );
						$featured_card_classes[] = 'wpex-' . $fc_bk . '-flex-shrink-0';
					}

					// Featured card bottom margin
					if ( empty( $this->atts['featured_divider'] ) || ! $is_featured_card_top ) {
						$fc_margin = $this->atts['featured_margin'] ? absint( $this->atts['featured_margin'] ) : 30;
						$featured_card_classes[] = 'wpex-mb-' . $fc_margin;
					}

					// Featured card side margin
					switch ( $featured_card_location ) {
						case 'left':
							$featured_card_classes[] = 'wpex-' . $fc_bk . '-mb-0';
							$featured_card_classes[] = 'wpex-' . $fc_bk . '-mr-' . $fc_margin;
							break;
						case 'right':
							$featured_card_classes[] = 'wpex-' . $fc_bk . '-mb-0';
							$featured_card_classes[] = 'wpex-' . $fc_bk . '-ml-' . $fc_margin;
							break;
					}

					// Featured card CSS.
					$featured_card_css = vcex_inline_style( array(
						'--wpex-card-media-max-width' => $this->atts['featured_media_max_width'] ?? null,
					) );

					// Display featured card.
					$inner_output .= '<div class="' . esc_attr( implode( ' ', $featured_card_classes ) ) . '"' . $featured_card_css . '>';

						if ( ! empty( $this->atts['featured_post_id'] ) ) {
							$featured_post_id = $this->atts['featured_post_id'];
							global $post;
							$post = get_post( $this->atts['featured_post_id'] );
							$inner_output .= wpex_get_card( $this->get_featured_card_args( $featured_post_id ) );
							wp_reset_postdata();
						} else {
							$count=0;
							while ( $this->query->have_posts() ) :
								$count++;
								if ( 2 === $count ) {
									break;
								}

								$this->query->the_post();

								$featured_post_id = get_the_ID();

								$inner_output .= wpex_get_card( $this->get_featured_card_args( $featured_post_id ) );

							endwhile;
						}

					$inner_output .= '</div>';

					if ( empty( $this->atts['featured_post_id'] ) ) {
						wp_reset_postdata();
						$this->query->rewind_posts();
					}

					if ( ! empty( $this->atts['featured_divider'] ) && $is_featured_card_top ) {
						$inner_output .= $this->featured_divider();
					}

				}

				/*-------------------------------------*/
				/* [ Entries start here ]
				/*-------------------------------------*/
				if ( $this->query->have_posts() ) :

					if ( $has_featured_card && ! $is_featured_card_top ) {
						$inner_output .= '<div class="wpex-post-cards-aside wpex-min-w-0 wpex-' . $fc_bk . '-flex-grow">';
					}

					$items_wrap_class = array();

					switch ( $display_type ) {

						case 'carousel':

							vcex_enqueue_carousel_scripts();

							// All carousels need a unique classname.
							$unique_classname = vcex_element_unique_classname();

							// Get carousel settings.
							$carousel_settings = vcex_get_carousel_settings( $this->atts, 'wpex_post_cards' );
							$carousel_css = vcex_get_carousel_inline_css( $unique_classname, $carousel_settings );

							$items_data['data-wpex-carousel'] = $carousel_settings;
							$items_wrap_class[] = 'wpex-posts-card-carousel';
							$items_wrap_class[] = 'wpex-carousel';
							$items_wrap_class[] = 'owl-carousel';

							if ( $carousel_css ) {
								$items_wrap_class[] = $unique_classname;
								$items_wrap_class[] = 'wpex-carousel--render-onload';
								$output .= $carousel_css;
							}

							// Flex carousel.
							if ( empty( $this->atts['auto_height'] ) || 'false' === $this->atts['auto_height'] ) {
								$items_wrap_class[] = 'wpex-carousel--flex';
							}

							// No margins.
							if ( array_key_exists( 'items_margin', $this->atts ) && empty( absint( $this->atts['items_margin'] ) ) ) {
								$items_wrap_class[] = 'wpex-carousel--no-margins';
							}

							// Arrow style.
							$arrows_style = ! empty( $this->atts['arrows_style'] ) ? $this->atts['arrows_style'] : 'default';
							$items_wrap_class[] = 'arrwstyle-' . sanitize_html_class( $arrows_style );

							// Arrow position.
							if ( ! empty( $this->atts['arrows_position'] ) && 'default' != $this->atts['arrows_position'] ) {
								$items_wrap_class[] = 'arrwpos-' . sanitize_html_class( $this->atts['arrows_position'] );
							}

							break;

						case 'list':
							$items_wrap_class[] = 'wpex-post-cards-list';
							$items_wrap_class[] = 'wpex-grid';
							if ( $grid_gap_class ) {
								$items_wrap_class[] = $grid_gap_class;
							}
							if ( vcex_validate_att_boolean( 'alternate_flex_direction', $this->atts ) ) {
								$items_wrap_class[] = 'wpex-post-cards-list--alternate-flex-direction';
							}
							if ( vcex_validate_att_boolean( 'list_divider_remove_last', $this->atts ) ) {
								$items_wrap_class[] = 'wpex-last-divider-none';
							}
							break;

						case 'grid':
						default:

							if ( 'css_grid' === $grid_style ) {
								$items_wrap_class[] = 'wpex-post-cards-grid wpex-grid';
								if ( $grid_is_responsive && ! empty( $this->atts['grid_columns_responsive_settings'] ) ) {
									$r_grid_columns = vcex_parse_multi_attribute( $this->atts['grid_columns_responsive_settings'] );
									if ( $r_grid_columns && is_array( $r_grid_columns ) ) {
										$r_grid_columns['d'] = $grid_columns;
										$grid_columns = $r_grid_columns;
									}
								}
								if ( $grid_is_responsive && function_exists( 'wpex_grid_columns_class' ) ) {
									$items_wrap_class[] = wpex_grid_columns_class( $grid_columns );
								} else {
									$items_wrap_class[] = 'wpex-grid-cols-' . sanitize_html_class( $grid_columns );
								}
							} else {
								$items_wrap_class[] = 'wpex-post-cards-grid';
								$items_wrap_class[] = 'wpex-row';
								$items_wrap_class[] = 'wpex-clr';
							}

							if ( 'masonry' === $grid_style ) {
								$items_wrap_class[] = 'wpex-masonry-grid';
								if ( function_exists( 'wpex_enqueue_masonry_scripts' ) ) {
									wpex_enqueue_masonry_scripts(); // uses theme masonry scripts.
								}
							}

							if ( $grid_gap_class ) {
								$items_wrap_class[] = $grid_gap_class;
							}

							break;

					} // end switch

					$inner_output .= '<div class="' . esc_attr( implode( ' ', $items_wrap_class ) ) . '"';

						// Add grid data attributes.
						if ( ! empty( $items_data ) ) {
							foreach ( $items_data as $key => $value ) {
								$inner_output .= ' ' . $key ."='" . esc_attr( $value ) . "'";
							}
						}

						// Add CSS to the items element.
						$grid_css_args = array();
						if ( ! empty( $this->atts['media_max_width'] ) ) {
							$grid_css_args['--wpex-card-media-max-width'] = $this->atts['media_max_width'];
						}
						if ( ! $grid_gap_class ) {
							switch( $display_type ) {
								case 'list':
									if ( ! empty( $this->atts['list_spacing'] ) ) {
										$grid_css_args['gap'] = $this->atts['list_spacing'];
									}
									break;
								case 'grid':
									if ( ! empty( $this->atts['grid_spacing'] ) ) {
										if ( 'css_grid' === $grid_style ) {
											$grid_css_args['gap'] = $this->atts['grid_spacing'];
										} else {
											$grid_css_args['--wpex-row-gap'] = $this->atts['grid_spacing'];
										}
									}
									break;
							}
						}
						if ( $grid_css_args ) {
							$inner_output .= vcex_inline_style( $grid_css_args );
						}

					$inner_output .= '>';

						// Add first divider if enabled.
						if ( 'list' === $display_type
							&& ! vcex_validate_att_boolean( 'list_divider_remove_first', $this->atts, true )
							&& ! $this->doing_loadmore
						) {
							$inner_output .= $this->list_divider( $this->atts );
						}

						// The Loop
						while ( $this->query->have_posts() ) :

							$this->query->the_post(); // !!! Important !!!

							$post_id = get_the_ID();
							$post_type = get_post_type( $post_id );
							$card_args['post_id'] = $post_id;

							if ( ! empty( $featured_post_id ) && empty( $this->atts['featured_post_id'] ) && $post_id === $featured_post_id ) {
								continue;
							}

							$entry_count++;

							$running_count++;
							set_query_var( 'wpex_loop_running_count', absint( $running_count ) );

							$item_class = array(
								'wpex-post-cards-entry',
								'post-' . sanitize_html_class( $post_id ),
								'type-' . sanitize_html_class( $post_type ),
							);

							switch ( $display_type ) {
								case 'carousel':
									$item_class[] = 'wpex-carousel-slide';
									break;
								case 'list':
									// No classes for list style.
									break;
								case 'grid':
								default:

									// Modern CSS grids.
									if ( 'css_grid' === $grid_style ) {
										$item_class[] = 'wpex-flex';
										$item_class[] = 'wpex-flex-col';
										$item_class[] = 'wpex-flex-grow';
									}

									// Old school grids.
									else {
										if ( $grid_is_responsive ) {
											$item_class[] = 'col';
										} else {
											$item_class[] = 'nr-col';
										}

										$item_class[] = 'col-' . sanitize_html_class( $entry_count );

										if ( $grid_columns ) {
											$item_class[] = 'span_1_of_' . sanitize_html_class( $grid_columns );
										}

										if ( $grid_is_responsive ) {
											$rs = vcex_parse_multi_attribute( $this->atts['grid_columns_responsive_settings'] );
											foreach ( $rs as $key => $val ) {
												if ( $val ) {
													$item_class[] = 'span_1_of_' . sanitize_html_class( $val ) . '_' . sanitize_html_class( $key );
												}
											}
										}
									}

									if ( 'masonry' === $grid_style ) {
										$item_class[] = 'wpex-masonry-col';
									}

									break;
							}

						if ( function_exists( 'vcex_get_post_term_classes' ) ) {
							$terms = vcex_get_post_term_classes();
							if ( $terms && is_array( $terms ) ) {
								foreach ( $terms as $term_name ) {
									$item_class[] = $term_name;
								}
							}
						}

						/**
						 * Filters the Post Cards shortcode entry classes.
						 *
						 * @param array $class
						 * @param array $shortcode_atts
						 */
						$item_class = (array) apply_filters( 'wpex_post_cards_entry_class', $item_class, $this->atts );

						// Begin entry output.
						$inner_output .= '<div class="' . esc_attr( implode( ' ', array_unique( $item_class ) ) ) . '">';
							$inner_output .= wpex_get_card( $card_args );
						$inner_output .= '</div>';

						// List Divider.
						if ( 'list' === $display_type && ! empty( $this->atts['list_divider'] ) ) {
							$inner_output .= $this->list_divider( $this->atts );
						}

						// Reset entry count.
						if ( 'grid' === $display_type
							&& 'fit_rows' === $grid_style
							&& $entry_count === $grid_columns
						) {
							$entry_count = 0;
						}

						endwhile;

					// Update ajax vars after loop.
					if ( isset( $ajax_atts ) ) {
						$ajax_atts['entry_count'] = $entry_count; // Update counter
						$ajax_atts['running_count'] = $running_count; // Update running count
					}

					// Reset post data.
					wp_reset_postdata();

					// Remove running count.
					set_query_var( 'wpex_loop_running_count', null );

				// Close element that holds main (not featured) posts.
				$inner_output .= '</div>';

				/*-------------------------------------*/
				/* [ Pagination ]
				/*-------------------------------------*/
				if ( 'grid' === $display_type || 'list' === $display_type ) {
					$pagination_type = $this->get_pagination_type();
					switch( $pagination_type ) {
						case 'loadmore';
						case 'infinite_scroll';
							if ( ! empty( $this->query->max_num_pages ) ) {
								vcex_loadmore_scripts();
								$ajax_atts = $ajax_atts ?? array();
								$infinite_scroll = ( 'infinite_scroll' === $pagination_type ) ? true : false;
								$inner_output .= vcex_get_loadmore_button( 'wpex_post_cards', $ajax_atts, $this->query, $infinite_scroll );
							}
							break;
						case 'numbered';
							$inner_output .= vcex_pagination( $this->query, false );
							break;
					}
				}

				// Close featured aside wrap.
				if ( $has_featured_card && ! $is_featured_card_top ) {
					$inner_output .= '</div>';
				}

				// Close featured flex wrap.
				if ( $has_featured_card && ! $is_featured_card_top ) {
					$inner_output .= '</div>';
				}

			endif; // has posts check.

			/*-------------------------------------*/
			/* [ Put inner_output inside wrap. ]
			/*-------------------------------------*/
			$output .= '<div';

					if ( ! empty( $this->atts['unique_id'] ) ) {
						$output .= ' id="' . esc_attr( $this->atts['unique_id'] ) . '"';
					}

					$output .= ' class="' . esc_attr( implode( ' ', $this->get_wrap_classes( $has_featured_card ) ) ) . '"';

					/*
					// @todo switch to this when adding ajax filter functionality.
					if ( isset( $ajax_atts ) ) {
						$output .= ' data-vcex-public-nonce="' . esc_attr( wp_create_nonce( 'vcex-public-nonce' ) ) . '"';
						$output .= ' data-vcex-settings="' . esc_attr( $this->get_json_data( $ajax_atts ) ) . '"';
					}*/

				$output .= '>';

			$output .= $inner_output;

			$output .= '</div>'; // close wrap

			// If no posts are found display message.
			else :

				$output .= vcex_no_posts_found_message( $this->atts );

			endif; // End post check.

			// Remove vars from memory.
			$this->atts = array();
			$this->query = null;
			$this->doing_loadmore = false;

			// Return shortcode output.
			return $output;
		}

		/**
		 * Return array of wrap classes.
		 */
		private function get_wrap_classes( $has_featured_card = false ) {
			$classes = array(
				'wpex-post-cards',
				'wpex-post-cards-' . sanitize_html_class( $this->atts['card_style'] ),
			);

			if ( $has_featured_card ) {
				$classes[] = 'wpex-post-cards-has-featured'; // @todo rename to use BEM
			}

			if ( ! empty( $this->atts['bottom_margin'] ) ) {
				$classes[] = vcex_sanitize_margin_class( $this->atts['bottom_margin'], 'wpex-mb-' );
			}

			if ( ! empty( $this->atts['el_class'] ) ) {
				$classes[] = wp_strip_all_tags( $this->atts['el_class'] );
			}

			if ( ! empty( $this->atts['css_animation'] ) && vcex_validate_att_boolean( 'css_animation_sequential', $this->atts ) ) {
				$classes[] = 'wpb-animate-in-sequence';
			}

			return $classes;
		}

		/**
		 * Return json data used for ajax functions.
		 */
		private function get_json_data( $ajax_atts = array() ) {
			return wp_json_encode( $ajax_atts, false );
		}

		/**
		 * Return card args based on shortcode atts.
		 */
		private function get_card_args() {
			$args = array(
				'style' => $this->atts['card_style'],
			);

			if ( ! empty( $this->atts['display_type'] ) ) {
				$args['display_type'] = $this->atts['display_type'];
			}

			if ( ! empty( $this->atts['link_type'] ) ) {
				$args['link_type'] = $this->atts['link_type'];
			}

			if ( ! empty( $this->atts['modal_title'] ) ) {
				$args['modal_title'] = $this->atts['modal_title'];
			}

			if ( ! empty( $this->atts['modal_template'] ) ) {
				$args['modal_template'] = $this->atts['modal_template'];
			}

			if ( ! empty( $this->atts['link_target'] ) ) {
				$args['link_target'] = $this->atts['link_target'];
			}

			if ( ! empty( $this->atts['link_rel'] ) ) {
				$args['link_rel'] = $this->atts['link_rel'];
			}

			if ( ! empty( $this->atts['title_font_size'] ) ) {
				$args['title_font_size'] = $this->atts['title_font_size'];
			}

			if ( ! empty( $this->atts['title_tag'] ) ) {
				$args['title_tag'] = $this->atts['title_tag'];
			}

			if ( ! empty( $this->atts['css_animation'] ) ) {
				$args['css_animation'] = $this->atts['css_animation'];
			}

			if ( isset( $this->atts['more_link_text'] ) && '' !== $this->atts['more_link_text'] ) {
				$args['more_link_text'] = $this->atts['more_link_text']; // allows "0" for disabling.
			}

			if ( ! empty( $this->atts['media_width'] ) ) {
				$args['media_width'] = $this->atts['media_width'];
			}

			if ( ! empty( $this->atts['media_breakpoint'] ) ) {
				$args['breakpoint'] = $this->atts['media_breakpoint'];
			}

			if ( empty( $this->atts['thumbnail_size'] ) || 'wpex_custom' === $this->atts['thumbnail_size'] ) {
				$args['thumbnail_size'] = array(
					$this->atts['thumbnail_width'],
					$this->atts['thumbnail_height'],
					$this->atts['thumbnail_crop'],
				);
			} else {
				$args['thumbnail_size'] = $this->atts['thumbnail_size'];
			}

			if ( ! empty( $this->atts['thumbnail_overlay_style'] ) ) {
				$args['thumbnail_overlay_style'] = $this->atts['thumbnail_overlay_style'];
			}

			if ( ! empty( $this->atts['thumbnail_overlay_button_text'] ) ) {
				$args['thumbnail_overlay_button_text'] = $this->atts['thumbnail_overlay_button_text'];
			}

			if ( ! empty( $this->atts['thumbnail_hover'] ) ) {
				$args['thumbnail_hover'] = $this->atts['thumbnail_hover'];
			}

			if ( ! empty( $this->atts['thumbnail_filter'] ) ) {
				$args['thumbnail_filter'] = $this->atts['thumbnail_filter'];
			}

			if ( ! empty( $this->atts['media_el_class'] ) ) {
				$args['media_el_class'] = $this->atts['media_el_class'];
			}

			if ( ! empty( $this->atts['card_el_class'] ) ) {
				$args['el_class'] = $this->atts['card_el_class'];
			}

			if ( isset( $this->atts['excerpt_length'] ) && '' !== $this->atts['excerpt_length'] ) {
				$args['excerpt_length'] = $this->atts['excerpt_length'];
			}

			if ( ! empty( $this->atts['alternate_flex_direction'] ) ) {
				$args['alternate_flex_direction'] = $this->atts['alternate_flex_direction'];
			}

			$allowed_media = $this->get_allowed_media();

			if ( ! is_null( $allowed_media ) ) {
				$args['allowed_media'] = $allowed_media;
			}

			return $args;
		}

		/**
		 * Check if loadmore is enabled.
		 */
		private function has_loadmore() {
			if ( ( ! empty( $this->atts['pagination'] ) && in_array( $this->atts['pagination'], array( 'loadmore', 'infinite_scroll' ) ) ) || vcex_validate_att_boolean( 'pagination_loadmore', $this->atts ) ) {
				return true;
			}
			return false;
		}

		/**
		 * Check if featured card is enabled.
		 */
		private function has_featured_card() {
			if ( $this->doing_loadmore ) {
				return false;
			}

			// Check if the featured card is enabled.
			$check = vcex_validate_att_boolean( 'featured_card', $this->atts );

			// Do not show featured card on paginated pages.
			if ( ( ! vcex_validate_att_boolean( 'featured_show_on_paged', $this->atts ) || $this->has_loadmore() ) && is_paged() ) {
				$check = false;
			}

			/**
			 * Filters whether the featured card is enabled or not.
			 *
			 * @param bool $check
			 */
			$check = (bool) apply_filters( 'wpex_post_cards_has_featured_card', $check, $this->atts );

			return $check;
		}

		/**
		 * Get supported media.
		 */
		private function get_allowed_media() {
			if ( array_key_exists( 'allowed_media', $this->atts ) ) {
				if ( $this->atts['allowed_media'] && is_string( $this->atts['allowed_media'] ) ) {
					$this->atts['allowed_media'] = wp_parse_list( $this->atts['allowed_media'] );
				}
				foreach ( $this->atts['allowed_media'] as $k => $v ) {
					if ( ! in_array( $v, array( 'thumbnail', 'video' ) ) ) {
						unset( $this->atts['allowed_media'][$k] );
					}
				}
				return $this->atts['allowed_media'];
			}
		}

		/**
		 * Check if featured card is enabled.
		 */
		private function has_featured_card_custom_query() {
			if ( $this->has_loadmore() || ! vcex_validate_att_boolean( 'featured_show_on_paged', $this->atts ) ) {
				return true;
			}
			return false;
		}

		/**
		 * Get featured card ID.
		 */
		private function get_featured_post_id() {
			$featured_post_id = 0;

			if ( ! empty( $this->atts['featured_post_id'] ) ) {
				$post_id = $this->atts['featured_post_id'];
				$post = get_post( $post_id );
				if ( $post && 'publish' === $post->post_status ) {
					$featured_post_id = wpex_parse_obj_id( $post_id, get_post_type( $post ) );
				}
			} elseif ( $this->has_featured_card_custom_query() ) {
				$query_args = $this->atts;
				$query_args['posts_per_page'] = 1;
				$query_posts = vcex_build_wp_query( $query_args );
				if ( $query_posts->have_posts() && ! empty( $query_posts->posts[0] ) ) {
					$featured_post_id = $query_posts->posts[0]->ID ?? 0;
				}
			}

			/**
			 * Filters the featured card ID.
			 *
			 * @param int $featured_post_id
			 * @param array $shortcode attributes
			 */
			$featured_post_id = (int) apply_filters( 'wpex_post_cards_featured_post_id', $featured_post_id, $this->atts );

			return $featured_post_id;
		}

		/**
		 * Featured card args.
		 *
		 * @todo can we optimize this to use a loop instead?
		 */
		private function get_featured_card_args( $post_id ) {
			$featured_card_style = ! empty( $this->atts['featured_style'] ) ? $this->atts['featured_style'] : $this->atts['card_style'];

			$args = array(
				'post_id'  => $post_id,
				'style'    => $featured_card_style,
				'featured' => true,
			);

			if ( ! empty( $this->atts['featured_title_font_size'] ) ) {
				$args['title_font_size'] = $this->atts['featured_title_font_size'];
			}

			if ( ! empty( $this->atts['featured_title_tag'] ) ) {
				$args['title_tag'] = $this->atts['featured_title_tag'];
			}

			if ( empty( $this->atts['featured_thumbnail_size'] ) || 'wpex_custom' === $this->atts['featured_thumbnail_size'] ) {
				$args['thumbnail_size'] = array(
					$this->atts['featured_thumbnail_width'],
					$this->atts['featured_thumbnail_height'],
					$this->atts['featured_thumbnail_crop'],
				);
			} else {
				$args['thumbnail_size'] = $this->atts['featured_thumbnail_size'];
			}

			if ( isset( $this->atts['featured_more_link_text'] ) && '' !== $this->atts['featured_more_link_text'] ) {
				$args['more_link_text'] = $this->atts['featured_more_link_text']; // allows "0" for disabling.
			}

			if ( isset( $this->atts['featured_excerpt_length'] ) && '' !== $this->atts['featured_excerpt_length'] ) {
				$args['excerpt_length'] = $this->atts['featured_excerpt_length'];
			}

			if ( ! empty( $this->atts['thumbnail_overlay_style'] ) ) {
				$args['thumbnail_overlay_style'] = $this->atts['thumbnail_overlay_style'];
			}

			if ( ! empty( $this->atts['thumbnail_overlay_button_text'] ) ) {
				$args['thumbnail_overlay_button_text'] = $this->atts['thumbnail_overlay_button_text'];
			}

			if ( ! empty( $this->atts['thumbnail_hover'] ) ) {
				$args['thumbnail_hover'] = $this->atts['thumbnail_hover'];
			}

			if ( ! empty( $this->atts['thumbnail_filter'] ) ) {
				$args['thumbnail_filter'] = $this->atts['thumbnail_filter'];
			}

			if ( ! empty( $this->atts['media_el_class'] ) ) {
				$args['media_el_class'] = $this->atts['media_el_class'];
			}

			if ( ! empty( $this->atts['featured_el_class'] ) ) {
				$args['el_class'] = $this->atts['featured_el_class'];
			}

			if ( ! empty( $this->atts['modal_title'] ) ) {
				$args['modal_title'] = $this->atts['modal_title'];
			}

			if ( ! empty( $this->atts['modal_template'] ) ) {
				$args['modal_template'] = $this->atts['modal_template'];
			}

			if ( ! empty( $this->atts['link_type'] ) ) {
				$args['link_type'] = $this->atts['link_type'];
			}

			if ( ! empty( $this->atts['link_target'] ) ) {
				$args['link_target'] = $this->atts['link_target'];
			}

			if ( ! empty( $this->atts['link_rel'] ) ) {
				$args['link_rel'] = $this->atts['link_rel'];
			}

			if ( ! empty( $this->atts['featured_media_width'] ) ) {
				$args['media_width'] = $this->atts['featured_media_width'];
			}

			if ( ! empty( $this->atts['featured_media_breakpoint'] ) ) {
				$args['breakpoint'] = $this->atts['featured_media_breakpoint'];
			} elseif ( ! empty( $this->atts['media_breakpoint'] ) ) {
				$args['breakpoint'] = $this->atts['media_breakpoint'];
			}

			$allowed_media = $this->get_allowed_media();

			if ( ! is_null( $allowed_media ) ) {
				$args['allowed_media'] = $allowed_media;
			}

			/**
			 * Filters the wpex_post_card featured card args.
			 *
			 * @param array $args
			 * @param array $shortcode_attributes
			 */
			$args = (array) apply_filters( 'wpex_post_cards_featured_card_args', $args, $this->atts );

			return $args;
		}

		/**
		 * Get featured card location.
		 */
		private function get_featured_card_location() {
			return $this->atts['featured_location'] ?? 'top';
		}

		/**
		 * Check if featured card is enabled.
		 */
		private function is_featured_card_top() {
			$location = $this->get_featured_card_location();
			if ( 'left' === $location || 'right' === $location ) {
				return false;
			}
			return true;
		}

		/**
		 * Featured Card Divider
		 */
		private function featured_divider() {
			$divider_class = array(
				'wpex-post-cards-featured-card-divider',
				'wpex-divider',
				'wpex-divider-' . sanitize_html_class( $this->atts['featured_divider'] ),
			);

			if ( ! empty( $this->atts['featured_divider_size'] ) ) {
				$divider_size = absint( $this->atts['featured_divider_size'] );
				if ( 1 === $divider_size ) {
					$divider_class[] = 'wpex-border-b';
				} else {
					$divider_class[] = 'wpex-border-b-' . $divider_size;
				}
			}

			$divider_style = '';

			if ( ! empty( $this->atts['featured_divider_color'] ) ) {
				$divider_style = vcex_inline_style( array(
					'border_color' => $this->atts['featured_divider_color'],
				) );
			}

			$spacing = ! empty( $this->atts['featured_divider_margin'] ) ? $this->atts['featured_divider_margin'] : 15;

			if ( ! empty( $this->atts['featured_margin'] ) ) {
				$divider_class[] = 'wpex-mt-' . sanitize_html_class( absint( $spacing ) );
				$divider_class[] = 'wpex-mb-' . sanitize_html_class( absint( $this->atts['featured_margin'] ) );
			} else {
				$divider_class[] = 'wpex-my-' . sanitize_html_class( absint( $spacing ) );
			}

			return '<div class="' . esc_attr( implode( ' ', $divider_class ) ) . '"' . $divider_style . '></div>';
		}

		/**
		 * List Divider.
		 */
		public function list_divider() {
			$divider_class = array(
				'wpex-card-list-divider',
				'wpex-divider',
				'wpex-divider-' . sanitize_html_class( $this->atts['list_divider'] ),
			);

			$divider_class[] = 'wpex-my-0'; // remove default margin since we want to use gaps.

			if ( ! empty( $this->atts['list_divider_size'] ) ) {
				$divider_size = absint( $this->atts['list_divider_size'] );
				if ( 1 === $divider_size ) {
					$divider_class[] = 'wpex-border-b';
				} else {
					$divider_class[] = 'wpex-border-b-' . $divider_size;
				}
			}

			$divider_style = '';

			if ( ! empty( $this->atts['list_divider_color'] ) ) {
				$divider_style = vcex_inline_style( array(
					'border_color' => $this->atts['list_divider_color'],
				) );
			}

			return '<div class="' . esc_attr( implode( ' ', $divider_class ) ) . '"' . $divider_style . '></div>';
		}

		/**
		 * Get display type.
		 */
		private function get_display_type() {
			return ! empty( $this->atts['display_type'] ) ? $this->atts['display_type'] : 'grid';
		}

		/**
		 * Get pagination type.
		 */
		private function get_pagination_type() {
			$pagination_type = $this->atts['pagination'] ?? '';
			$is_auto_query = vcex_validate_att_boolean( 'auto_query', $this->atts );

			$allowed_choices = array(
				'loadmore',
				'numbered',
				'infinite_scroll',
			);

			// Allowed pagination styles.
			if ( ! in_array( $pagination_type, $allowed_choices ) ) {
				$pagination_type = '';
			}

			// Enable pagination for auto and custom queries.
			if ( ! $pagination_type && ( $is_auto_query || ( vcex_validate_att_boolean( 'custom_query', $this->atts ) && ! empty( $this->query->query['pagination'] ) ) ) ) {
					$pagination_type = 'numbered';
			}

			// relevanssi fix.
			if ( $pagination_type && $is_auto_query && function_exists( 'relevanssi_do_query' ) && is_search() ) {
				$pagination_type = 'numbered';
			}

			return $pagination_type;
		}

		/**
		 * Get grid style.
		 */
		private function get_grid_style() {
			$allowed = array(
				'css_grid',
				'masonry',
				'fit_rows',
			);
			if ( ! empty( $this->atts['grid_style'] ) && in_array( $this->atts['grid_style'], $allowed ) ) {
				return $this->atts['grid_style'];
			}
			return 'fit_rows';
		}

		/**
		 * Get grid style.
		 */
		private function get_grid_columns() {
			return ! empty( $this->atts['grid_columns'] ) ? absint( $this->atts['grid_columns'] ) : 1;
		}

		/**
		 * Get grid gap class.
		 */
		private function get_grid_gap_class() {
			$display_type = $this->get_display_type();
			if ( 'carousel' === $display_type ) {
				return;
			}
			$gap        = '';
			$grid_style = $this->get_grid_style();
			switch( $display_type ) {
				case 'list':
					$gap = ! empty( $this->atts['list_spacing'] ) ? $this->atts['list_spacing'] : '15';
					break;
				case 'grid':
					$default = '';
					if ( 'css_grid' === $grid_style ) {
						$default = '20';  // css_grid needs a default gap.
					}
					$gap = ! empty( $this->atts['grid_spacing'] ) ? $this->atts['grid_spacing'] : $default;
					break;
			}
			if ( $gap ) {
				if ( 'list' === $display_type || ( 'grid' === $display_type && 'css_grid' === $grid_style ) ) {
					$use_utl_class = true;
				} else {
					$use_utl_class = false;
				}
				if ( 'none' === $gap ) {
					if ( $use_utl_class ) {
						return 'wpex-gap-0';
					} else {
						return 'gap-none';
					}
				}
				if ( function_exists( 'wpex_column_gaps' )
					&& array_key_exists( absint( $gap ), wpex_column_gaps() )
				) {
					if ( $use_utl_class ) {
						return 'wpex-gap-' . absint( $gap );
					} else {
						return 'gap-' . absint( $gap );
					}
				}
			}
		}

		/**
		 * Returns array of breakpoint choices for the media element.
		 */
		public static function get_media_breakpoint_choices() {
			$choices = (array) vcex_breakpoint_choices();
			if ( $choices ) {
				$choices[esc_html__( 'Do not stack', 'total-theme-core' )] = 'false';
			}
			return $choices;
		}

		/**
		 * Array of shortcode parameters.
		 */
		public static function get_params() {
			$params = array(
				// General
				array(
					'type' => 'vcex_wpex_card_select',
					'heading' => esc_html__( 'Card Style', 'total-theme-core' ),
					'param_name' => 'card_style',
					'description' => esc_html__( 'Select your card style. Note: Not all settings are used for every card style', 'total-theme-core' ) . ' ' . sprintf( esc_html__( '%sPreview card styles%s', 'total-theme-core' ), '<a href="https://total.wpexplorer.com/features/cards/" target="_blank" rel="noopener noreferrer">', '</a>' ),
					'admin_label' => true,
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Bottom Margin', 'total-theme-core' ),
					'param_name' => 'bottom_margin',
					'value' => vcex_margin_choices(),
					'admin_label' => true,
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Display Type', 'total-theme-core' ),
					'param_name' => 'display_type',
					'value' => array(
						esc_html__( 'Grid', 'total-theme-core' ) => 'grid',
						esc_html__( 'List', 'total-theme-core' ) => 'list',
						esc_html__( 'Carousel', 'total-theme-core' ) => 'carousel',
					),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Grid Style', 'total-theme-core' ),
					'param_name' => 'grid_style',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => 'fit_rows',
						esc_html__( 'Masonry', 'total-theme-core' ) => 'masonry',
						esc_html__( 'Modern CSS Grid', 'total-theme-core' ) => 'css_grid',
					),
					'edit_field_class' => 'vc_col-sm-4 vc_column clear',
					'dependency' => array( 'element' => 'display_type', 'value' => array( 'grid', 'masonry_grid' ) ),
				),
				array(
					'type' => 'vcex_grid_columns',
					'heading' => esc_html__( 'Columns', 'total-theme-core' ),
					'param_name' => 'grid_columns',
					'std' => '3',
					'edit_field_class' => 'vc_col-sm-4 vc_column',
					'dependency' => array( 'element' => 'display_type', 'value' => array( 'grid', 'masonry_grid' ) ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Responsive', 'total-theme-core' ),
					'param_name' => 'grid_columns_responsive',
					'value' => array(
						esc_html__( 'Yes', 'total-theme-core' ) => 'true',
						esc_html__( 'No', 'total-theme-core' ) => 'false',
					),
					'edit_field_class' => 'vc_col-sm-4 vc_column',
					'dependency' => array( 'element' => 'grid_columns', 'value' => array( '2', '3', '4', '5', '6', '7', '8', '9', '10' ) ),
				),
				array(
					'type' => 'vcex_grid_columns_responsive',
					'heading' => esc_html__( 'Responsive Column Settings', 'total-theme-core' ),
					'param_name' => 'grid_columns_responsive_settings',
					'dependency' => array( 'element' => 'grid_columns_responsive', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_preset_textfield',
					'heading' => esc_html__( 'Column Gap', 'total-theme-core' ),
					'param_name' => 'grid_spacing',
					'choices' => 'gap',
					'dependency' => array( 'element' => 'display_type', 'value' => array( 'grid', 'masonry_grid' ) ),
				),
				array(
					'type' => 'vcex_preset_textfield',
					'heading' => esc_html__( 'List Spacing', 'total-theme-core' ),
					'param_name' => 'list_spacing',
					'choices' => 'gap',
					'dependency' => array( 'element' => 'display_type', 'value' => 'list' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'List Divider', 'total-theme-core' ),
					'param_name' => 'list_divider',
					'value' => array(
						esc_html__( 'None', 'total-theme-core' ) => '',
						esc_html__( 'Solid', 'total-theme-core' ) => 'solid',
						esc_html__( 'Dashed', 'total-theme-core' ) => 'dashed',
						esc_html__( 'Dotted', 'total-theme-core' ) => 'dotted',
					),
					'dependency' => array( 'element' => 'display_type', 'value' => 'list' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'List Divider Size', 'total-theme-core' ),
					'param_name' => 'list_divider_size',
					'dependency' => array( 'element' => 'list_divider', 'not_empty' => true ),
					'value' => vcex_border_width_choices(),
				),
				array(
					'type' => 'vcex_colorpicker',
					'heading' => esc_html__( 'List Divider Color', 'total-theme-core' ),
					'param_name' => 'list_divider_color',
					'dependency' => array( 'element' => 'list_divider', 'not_empty' => true ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Remove Divider Before First Entry?', 'total-theme-core' ),
					'param_name' => 'list_divider_remove_first',
					'std' => 'true',
					'dependency' => array( 'element' => 'list_divider', 'not_empty' => true ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Remove Divider After Last Entry?', 'total-theme-core' ),
					'param_name' => 'list_divider_remove_last',
					'std' => 'false',
					'dependency' => array( 'element' => 'list_divider', 'not_empty' => true ),
				),
				// General
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Alternate Thumbnail Position', 'total-theme-core' ),
					'param_name' => 'alternate_flex_direction',
					'description' => esc_html__( 'Enable to alternate the position of your thumbnail when using certain cards styles. For example if you are using a card style with a left thumbnail every other item will display the thumbnail on the right.', 'total-theme-core' ),
					'std' => 'false',
					'dependency' => array( 'element' => 'display_type', 'value' => 'list' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Element ID', 'total-theme-core' ),
					'param_name' => 'unique_id',
					'admin_label' => true,
					'description' => vcex_shortcode_param_description( 'unique_id' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Extra class name', 'total-theme-core' ),
					'description' => vcex_shortcode_param_description( 'el_class' ),
					'param_name' => 'el_class',
					'admin_label' => true,
				),
				vcex_vc_map_add_css_animation(),
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Sequential Animation', 'total-theme-core' ),
					'param_name' => 'css_animation_sequential',
				),
				// Query
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Automatic Query', 'total-theme-core' ),
					'param_name' => 'auto_query',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Enable to display items from the current query. For use when overriding an archive (such as categories) with a template.', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'std' => 'post',
					'heading' => esc_html__( 'Automatic Query Preview Post Type', 'total-theme-core' ),
					'param_name' => 'auto_query_preview_pt',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Enter a post type name to use as the placeholder for the preview while editing in the WPBakery live editor.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'auto_query', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Advanced Query', 'total-theme-core' ),
					'param_name' => 'custom_query',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Enable to build a custom query using your own parameters.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'auto_query', 'value' => 'false' ),
				),
				array(
					'type' => 'textarea_safe',
					'heading' => esc_html__( 'Query Parameter String or Callback Function Name', 'total-theme-core' ),
					'param_name' => 'custom_query_args',
					'description' => vcex_shortcode_param_description( 'advanced_query' ),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => array( 'true' ) ),
				),
				array(
					'type' => 'posttypes', // @todo update to allow attachments post type.
					'heading' => esc_html__( 'Post types', 'total-theme-core' ),
					'param_name' => 'post_types',
					'std' => 'post',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'admin_label' => true,
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Posts Per Page', 'total-theme-core' ),
					'param_name' => 'posts_per_page',
					'value' => '12',
					'description' => esc_html__( 'You can enter "-1" to display all posts.', 'total-theme-core' ),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Pagination', 'total-theme-core' ),
					'param_name' => 'pagination',
					'value' => array(
						esc_html__( 'Disabled', 'total-theme-core' ) => '',
						esc_html__( 'Numbered', 'total-theme-core' ) => 'numbered',
						esc_html__( 'Load More', 'total-theme-core' ) => 'loadmore',
						esc_html__( 'Infinite Scroll', 'total-theme-core' ) => 'infinite_scroll',
					),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'By default pagination is disabled unless using an Auto Query in which case it will used the numbered pagination by default.', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Sticky Posts Only', 'total-theme-core' ),
					'param_name' => 'show_sticky_posts',
					'std' => 'false',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => array( 'false' ) ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Ignore Sticky Posts', 'total-theme-core' ),
					'param_name' => 'ignore_sticky_posts',
					'std' => 'false',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Include sticky posts, but not at the top.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => array( 'false' ) ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Exclude Sticky Posts', 'total-theme-core' ),
					'param_name' => 'exclude_sticky_posts',
					'std' => 'false',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Remove sticky posts completely.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => array( 'false' ) ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Post With Thumbnails Only', 'total-theme-core' ),
					'param_name' => 'thumbnail_query',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Offset', 'total-theme-core' ),
					'param_name' => 'offset',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Number of post to displace or pass over.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'autocomplete',
					'heading' => esc_html__( 'Query Specific Posts', 'total-theme-core' ),
					'param_name' => 'posts_in',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'settings' => array(
						'multiple' => true,
						'min_length' => 1,
						'groups' => false,
						'unique_values' => true,
						'display_inline' => true,
						'delay' => 0,
						'auto_focus' => true,
					),
					'description' => esc_html__( 'Start typing a post name to locate and add it. Make sure you have selected the Post Types above so they match the post types of the selected posts.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'autocomplete',
					'heading' => esc_html__( 'Limit By Author', 'total-theme-core' ),
					'param_name' => 'author_in',
					'settings' => array(
						'multiple' => true,
						'min_length' => 1,
						'groups' => false,
						'unique_values' => true,
						'display_inline' => true,
						'delay' => 0,
						'auto_focus' => true,
					),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Query by Taxonomy', 'total-theme-core' ),
					'param_name' => 'tax_query',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'autocomplete',
					'heading' => esc_html__( 'Taxonomy Name', 'total-theme-core' ),
					'param_name' => 'tax_query_taxonomy',
					'settings' => array(
						'multiple' => false,
						'min_length' => 1,
						'groups' => false,
						'display_inline' => true,
						'delay' => 0,
						'auto_focus' => true,
					),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'If you do not see your taxonomy in the dropdown you can still enter the taxonomy name manually.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'tax_query', 'value' => 'true' ),
				),
				array(
					'type' => 'autocomplete',
					'heading' => esc_html__( 'Terms', 'total-theme-core' ),
					'param_name' => 'tax_query_terms',
					'settings' => array(
						'multiple' => true,
						'min_length' => 1,
						'groups' => true,
						'display_inline' => true,
						'delay' => 0,
						'auto_focus' => true,
					),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'If you do not see your terms in the dropdown you can still enter the term slugs manually seperated by a space.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'tax_query', 'value' => 'true' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Order', 'total-theme-core' ),
					'param_name' => 'order',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => 'default',
						esc_html__( 'DESC', 'total-theme-core' ) => 'DESC',
						esc_html__( 'ASC', 'total-theme-core' ) => 'ASC',
					),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Order By', 'total-theme-core' ),
					'param_name' => 'orderby',
					'value' => vcex_orderby_array(),
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Orderby: Meta Key', 'total-theme-core' ),
					'param_name' => 'orderby_meta_key',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'dependency' => array( 'element' => 'orderby', 'value' => array( 'meta_value_num', 'meta_value' ) ),
				),
				array(
					'type' => 'textarea',
					'heading' => esc_html__( 'No Posts Found Message', 'total-theme-core' ),
					'param_name' => 'no_posts_found_message',
					'group' => esc_html__( 'Query', 'total-theme-core' ),
					'description' => esc_html__( 'Leave empty to disable.', 'total-theme-core' ),
				),
				// Entry
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Title Font Size', 'total-theme-core' ),
					'param_name' => 'title_font_size',
					'value' => vcex_font_size_choices(),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Title Tag', 'total-theme-core' ),
					'param_name' => 'title_tag',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => '',
						'h2' => 'h2',
						'h3' => 'h3',
						'h4' => 'h4',
						'h5' => 'h5',
						'h6' => 'h6',
						'div' => 'div',
					),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Media Width', 'total-theme-core' ),
					'param_name' => 'media_width',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => '',
						'20%'  => '20',
						'25%'  => '25',
						'30%'  => '30',
						'33%'  => '33',
						'40%'  => '40',
						'50%'  => '50',
						'60%'  => '60',
						'70%'  => '70',
						'80%'  => '80',
					),
					'description' => esc_html__( 'Applies to card styles that have the media (image/video) displayed to the side.', 'total-theme-core' ),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Media Max-Width', 'total-theme-core' ),
					'param_name' => 'media_max_width',
					'description' => esc_html__( 'Allows you to set a max-width for the media element. For example if you select 60% above for the media width but want to make sure the image is never larger than 200px wide you can enter 200px here.', 'total-theme-core' ),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Breakpoint', 'total-theme-core' ),
					'param_name' => 'media_breakpoint',
					'value' => self::get_media_breakpoint_choices(),
					'description' => esc_html__( 'The breakpoint at which a left/right card styles swaps to a column view. The default for most cards is "md".', 'total-theme-core' ),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_image_sizes',
					'heading' => esc_html__( 'Thumbnail Size', 'total-theme-core' ),
					'param_name' => 'thumbnail_size',
					'std' => 'full',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'description' => esc_html__( 'Note: For security reasons custom cropping only works on images hosted on your own server in the WordPress uploads folder. If you are using an external image it will display in full.', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_image_crop_locations',
					'heading' => esc_html__( 'Thumbnail Crop Location', 'total-theme-core' ),
					'param_name' => 'thumbnail_crop',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'dependency' => array( 'element' => 'thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Thumbnail Crop Width', 'total-theme-core' ),
					'param_name' => 'thumbnail_width',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'dependency' => array( 'element' => 'thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Thumbnail Crop Height', 'total-theme-core' ),
					'param_name' => 'thumbnail_height',
					'description' => esc_html__( 'Leave empty to disable vertical cropping and keep image proportions.', 'total-theme-core' ),
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'dependency' => array( 'element' => 'thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Excerpt Length', 'total-theme-core' ),
					'param_name' => 'excerpt_length',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'description' => esc_html__( 'Enter how many words to display for the excerpt. To display the full post content enter "-1". To display the full post content up to the "more" tag enter "9999".', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Custom More Link Text', 'total-theme-core' ),
					'param_name' => 'more_link_text',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'description' => esc_html__( 'You can enter "0" to disable.', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Extra class name', 'total-theme-core' ),
					'param_name' => 'card_el_class',
					'group' => esc_html__( 'Entry', 'total-theme-core' ),
					'description' => esc_html__( 'Extra class name to apply to the ".wpex-card" element.', 'total-theme-core' ),
				),
				// Carousel Settings
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Arrows', 'total-theme-core' ),
					'param_name' => 'arrows',
					'std' => 'true',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_carousel_arrow_styles',
					'heading' => esc_html__( 'Arrows Style', 'total-theme-core' ),
					'param_name' => 'arrows_style',
					'dependency' => array( 'element' => 'arrows', 'value' => 'true' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_carousel_arrow_positions',
					'heading' => esc_html__( 'Arrows Position', 'total-theme-core' ),
					'param_name' => 'arrows_position',
					'dependency' => array( 'element' => 'arrows', 'value' => 'true' ),
					'std' => 'default',
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Dot Navigation', 'total-theme-core' ),
					'param_name' => 'dots',
					'std' => 'false',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Auto Play', 'total-theme-core' ),
					'param_name' => 'auto_play',
					'std' => 'false',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Autoplay interval timeout.', 'total-theme-core' ),
					'param_name' => 'timeout_duration',
					'value' => '5000',
					'description' => esc_html__( 'Time in milliseconds between each auto slide. Default is 5000.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'auto_play', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Infinite Loop', 'total-theme-core' ),
					'param_name' => 'infinite_loop',
					'std' => 'true',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Center Item', 'total-theme-core' ),
					'param_name' => 'center',
					'std' => 'false',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Animation Speed', 'total-theme-core' ),
					'param_name' => 'animation_speed',
					'value' => '250',
					'description' => esc_html__( 'Default is 250 milliseconds. Enter 0.0 to disable.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Items To Display', 'total-theme-core' ),
					'param_name' => 'items',
					'value' => '4',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'vcex_ofswitch',
					'std' => 'false',
					'heading' => esc_html__( 'Auto Height?', 'total-theme-core' ),
					'param_name' => 'auto_height',
					'dependency' => array( 'element' => 'items', 'value' => '1' ),
					'description' => esc_html__( 'Allows the carousel to change height based on the active item. This setting is used only when you are displaying 1 item per slide.', 'total-theme-core' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Items To Scrollby', 'total-theme-core' ),
					'param_name' => 'items_scroll',
					'value' => '1',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Tablet: Items To Display', 'total-theme-core' ),
					'param_name' => 'tablet_items',
					'value' => '3',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Mobile Landscape: Items To Display', 'total-theme-core' ),
					'param_name' => 'mobile_landscape_items',
					'value' => '2',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Mobile Portrait: Items To Display', 'total-theme-core' ),
					'param_name' => 'mobile_portrait_items',
					'value' => '1',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Margin Between Items', 'total-theme-core' ),
					'param_name' => 'items_margin',
					'value' => '15',
					'dependency' => array( 'element' => 'display_type', 'value' => 'carousel' ),
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' )
				),
				// Media
				array(
					'type' => 'checkbox',
					'heading' => esc_html__( 'Allowed Media Types', 'total-theme-core' ),
					'param_name' => 'allowed_media',
					'std' => 'thumbnail',
					'value' => array(
						esc_html__( 'Thumbnail', 'js_composer' ) => 'thumbnail',
						esc_html__( 'Video', 'js_composer' ) => 'video',
					//	esc_html__( 'Audio', 'js_composer' ) => 'audio',
					//	esc_html__( 'Gallery', 'js_composer' ) => 'gallery',
					),
					'description' => esc_html__( 'Note: Not all card styles support all media types.', 'total-theme-core' ),
					'group' => esc_html__( 'Media', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_overlay',
					'heading' => esc_html__( 'Thumbnail Overlay', 'total-theme-core' ),
					'param_name' => 'thumbnail_overlay_style',
					'group' => esc_html__( 'Media', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Overlay Button Text', 'total-theme-core' ),
					'param_name' => 'thumbnail_overlay_button_text',
					'group' => esc_html__( 'Media', 'total-theme-core' ),
					'dependency' => array( 'element' => 'thumbnail_overlay_style', 'value' => 'hover-button' ),
				),
				array(
					'type' => 'vcex_image_hovers',
					'heading' => esc_html__( 'Image Hover', 'total-theme-core' ),
					'param_name' => 'thumbnail_hover',
					'group' => esc_html__( 'Media', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_image_filters',
					'heading' => esc_html__( 'Image Filter', 'total-theme-core' ),
					'param_name' => 'thumbnail_filter',
					'group' => esc_html__( 'Media', 'total-theme-core' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Extra class name', 'total-theme-core' ),
					'param_name' => 'media_el_class',
					'description' => esc_html__( 'Extra class name to apply to the ".wpex-card-thumbnail" element.', 'total-theme-core' ),
					'group' => esc_html__( 'Media', 'total-theme-core' ),
				),
				// Link
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Link Type', 'total-theme-core' ),
					'param_name' => 'link_type',
					'group' => esc_html__( 'Link', 'total-theme-core' ),
					'value' => array(
						esc_html__( 'Link to post', 'total-theme-core' ) => '',
						esc_html__( 'Lightbox', 'total-theme-core' ) => 'lightbox',
						esc_html__( 'Modal Popup', 'total-theme-core' ) => 'modal',
						esc_html__( 'None', 'total-theme-core' ) => 'none',
					),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Modal Title', 'total-theme-core' ),
					'param_name' => 'modal_title',
					'std' => 'true',
					'dependency' => array( 'element' => 'link_type', 'value' => 'modal' ),
					'group' => esc_html__( 'Link', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_template_select',
					'heading' => esc_html__( 'Modal Template', 'total-theme-core' ),
					'param_name' => 'modal_template',
					'dependency' => array( 'element' => 'link_type', 'value' => 'modal' ),
					'group' => esc_html__( 'Link', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Link Target', 'total-theme-core' ),
					'param_name' => 'link_target',
					'group' => esc_html__( 'Link', 'total-theme-core' ),
					'value' => array(
						esc_html__( 'Same Tab', 'total-theme-core' ) => '',
						esc_html__( 'New Tab', 'total-theme-core' ) => '_blank',
					),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Link Rel', 'total-theme-core' ),
					'param_name' => 'link_rel',
					'group' => esc_html__( 'Link', 'total-theme-core' ),
				),
				// Featured
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Featured Card', 'total-theme-core' ),
					'param_name' => 'featured_card',
					'std' => 'false',
					'description' => esc_html__( 'Enable to display the first entry as a "featured" card with it\'s own unique style above the other entries.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_wpex_card_select',
					'heading' => esc_html__( 'Featured Card Style', 'total-theme-core' ),
					'param_name' => 'featured_style',
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'autocomplete',
					'heading' => esc_html__( 'Featured Post', 'total-theme-core' ),
					'param_name' => 'featured_post_id',
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'settings' => array(
						'multiple' => false,
						'min_length' => 1,
						'groups' => false,
						'unique_values' => true,
						'display_inline' => true,
						'delay' => 0,
						'auto_focus' => true,
					),
					'description' => esc_html__( 'Start typing a post name to locate and select it. Leave empty to display the first post as the featured post.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Display on Paginated Pages', 'total-theme-core' ),
					'param_name' => 'featured_show_on_paged',
					'std' => 'true',
					'description' => esc_html__( 'If disabled your posts per page count does not need to include the featured post.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Placement', 'total-theme-core' ),
					'param_name' => 'featured_location',
					'value' => array(
						esc_html__( 'Top', 'total-theme-core' ) => 'top',
						esc_html__( 'Left', 'total-theme-core' ) => 'left',
						esc_html__( 'Right', 'total-theme-core' ) => 'right',
					),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Width', 'total-theme-core' ),
					'param_name' => 'featured_width',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => '',
						'70%' => '70%',
						'67%' => '67%',
						'60%' => '60%',
						'50%' => '50%',
					),
					'dependency' => array( 'element' => 'featured_location', 'value' => array( 'left', 'right' ) ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Breakpoint', 'total-theme-core' ),
					'param_name' => 'featured_breakpoint',
					'value' => vcex_breakpoint_choices(),
					'dependency' => array( 'element' => 'featured_location', 'value' => array( 'left', 'right' ) ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Margin', 'total-theme-core' ),
					'param_name' => 'featured_margin',
					'value' => vcex_margin_choices(),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Divider Style', 'total-theme-core' ),
					'param_name' => 'featured_divider',
					'value' => array(
						esc_html__( 'None', 'total-theme-core' ) => '',
						esc_html__( 'Solid', 'total-theme-core' ) => 'solid',
						esc_html__( 'Dashed', 'total-theme-core' ) => 'dashed',
						esc_html__( 'Dotted', 'total-theme-core' ) => 'dotted',
					),
					'dependency' => array( 'element' => 'featured_location', 'value' => 'top' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Divider Size', 'total-theme-core' ),
					'param_name' => 'featured_divider_size',
					'value' => vcex_border_width_choices(),
					'dependency' => array( 'element' => 'featured_divider', 'not_empty' => true ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Divider Margin', 'total-theme-core' ),
					'param_name' => 'featured_divider_margin',
					'value' => vcex_margin_choices(),
					'dependency' => array( 'element' => 'featured_divider', 'not_empty' => true ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'vcex_colorpicker',
					'heading' => esc_html__( 'List Divider Color', 'total-theme-core' ),
					'param_name' => 'featured_divider_color',
					'dependency' => array( 'element' => 'featured_divider', 'not_empty' => true ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Title Font Size', 'total-theme-core' ),
					'param_name' => 'featured_title_font_size',
					'value' => vcex_font_size_choices(),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Title Tag', 'total-theme-core' ),
					'param_name' => 'featured_title_tag',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => '',
						'h2' => 'h2',
						'h3' => 'h3',
						'h4' => 'h4',
						'h5' => 'h5',
						'h6' => 'h6',
						'div' => 'div',
					),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Media Width', 'total-theme-core' ),
					'param_name' => 'featured_media_width',
					'value' => array(
						esc_html__( 'Default', 'total-theme-core' ) => '',
						'20%'  => '20',
						'25%'  => '25',
						'30%'  => '30',
						'33%'  => '33',
						'40%'  => '40',
						'50%'  => '50',
						'60%'  => '60',
						'70%'  => '70',
						'80%'  => '80',
					),
					'description' => esc_html__( 'Applies to card styles that have the media (image/video) displayed to the side.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Media Max-Width', 'total-theme-core' ),
					'param_name' => 'featured_media_max_width',
					'description' => esc_html__( 'Allows you to set a max-width for the media element. For example if you select 60% above for the media width but want to make sure the image is never larger than 200px wide you can enter 200px here.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Breakpoint', 'total-theme-core' ),
					'param_name' => 'featured_media_breakpoint',
					'value' => self::get_media_breakpoint_choices(),
					'description' => esc_html__( 'The breakpoint at which a left/right card styles swaps to a column view. The default for most cards is "md".', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_image_sizes',
					'heading' => esc_html__( 'Thumbnail Size', 'total-theme-core' ),
					'param_name' => 'featured_thumbnail_size',
					'std' => 'full',
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'description' => esc_html__( 'Note: For security reasons custom cropping only works on images hosted on your own server in the WordPress uploads folder. If you are using an external image it will display in full.', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'vcex_image_crop_locations',
					'heading' => esc_html__( 'Thumbnail Crop Location', 'total-theme-core' ),
					'param_name' => 'featured_thumbnail_crop',
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Thumbnail Crop Width', 'total-theme-core' ),
					'param_name' => 'featured_thumbnail_width',
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Thumbnail Crop Height', 'total-theme-core' ),
					'param_name' => 'featured_thumbnail_height',
					'description' => esc_html__( 'Leave empty to disable vertical cropping and keep image proportions.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_thumbnail_size', 'value' => 'wpex_custom' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Excerpt Length', 'total-theme-core' ),
					'param_name' => 'featured_excerpt_length',
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'description' => esc_html__( 'Enter how many words to display for the excerpt. To display the full post content enter "-1". To display the full post content up to the "more" tag enter "9999".', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Custom More Link Text', 'total-theme-core' ),
					'param_name' => 'featured_more_link_text',
					'description' => esc_html__( 'You can enter "0" to disable.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Extra class name', 'total-theme-core' ),
					'param_name' => 'featured_el_class',
					'description' => esc_html__( 'Extra class name to apply to the ".wpex-card" element.', 'total-theme-core' ),
					'group' => esc_html__( 'Featured', 'total-theme-core' ),
					'dependency' => array( 'element' => 'featured_card', 'value' => 'true' ),
				),
				// Hidden fields
				array( 'type' => 'hidden', 'param_name' => 'query_vars' ), // used for AJAX - !! important !!
				array( 'type' => 'hidden', 'param_name' => 'pagination_loadmore' ),
			);

			/*
			@todo Enable when we add Card styles for WooCommerce
			array(
				'type' => 'vcex_ofswitch',
				'std' => 'false',
				'heading' => esc_html__( 'Featured Products Only', 'total-theme-core' ),
				'param_name' => 'featured_products_only',
				'group' => esc_html__( 'Query', 'total-theme-core' ),
				'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
			),
			array(
				'type' => 'vcex_ofswitch',
				'std' => 'false',
				'heading' => esc_html__( 'Exclude Out of Stock Products', 'total-theme-core' ),
				'param_name' => 'exclude_products_out_of_stock',
				'group' => esc_html__( 'Query', 'total-theme-core' ),
				'dependency' => array( 'element' => 'custom_query', 'value' => 'false' ),
			),*/

			/**
			 * Filters the wpex_post_card shortcode parameters.
			 *
			 * @param array $params
			 * @param string shortcode_tag | wpex_post_cards
			 */
			$params = (array) apply_filters( 'vcex_shortcode_params', $params, 'wpex_post_cards' );

			return $params;
		}

		/**
		 * Parses deprecated params.
		 */
		public static function parse_deprecated_attributes( $atts = '' ) {
			if ( empty( $atts ) || ! is_array( $atts ) ) {
				return $atts;
			}

			if ( isset( $atts['pagination_loadmore'] ) && vcex_validate_att_boolean( 'pagination_loadmore', $atts ) ) {
				$atts['pagination'] = 'loadmore';
				unset( $atts['pagination_loadmore'] );
			} elseif ( isset( $atts['pagination'] ) ) {
				switch ( $atts['pagination'] ) {
					case 'true':
						$atts['pagination'] = 'numbered';
						break;
					case 'false':
						$atts['pagination'] = '';
						break;
				}
			}

			return $atts;
		}

	}

}
new WPEX_Post_Cards_Shortcode;

if ( class_exists( 'WPBakeryShortCode' ) && ! class_exists( 'WPBakeryShortCode_wpex_post_cards' ) ) {
	class WPBakeryShortCode_wpex_post_cards extends WPBakeryShortCode {}
}