<?php
namespace TotalTheme\CPT;

defined( 'ABSPATH' ) || exit;

/**
 * Custom Post Type Single Blocks.
 *
 * @package TotalTheme
 * @subpackage CPT
 * @version 5.4.2
 */
class Single_Blocks {

	/**
	 * Returns the array of custom post type single blocks.
	 */
	public static function choices() {
		$choices = array(
			'media'       => esc_html__( 'Media (Thumbnail, Slider, Video)', 'total' ),
			'title'       => esc_html__( 'Title', 'total' ),
			'meta'        => esc_html__( 'Meta', 'total' ),
			'post-series' => esc_html__( 'Post Series', 'total' ),
			'content'     => esc_html__( 'Content', 'total' ),
			'page-links'  => esc_html__( 'Page Links', 'total' ),
			'share'       => esc_html__( 'Social Share', 'total' ),
			'author-bio'  => esc_html__( 'Author Bio', 'total' ),
			'related'     => esc_html__( 'Related', 'total' ),
			'comments'    => esc_html__( 'Comments', 'total' ),
		);

		/**
		 * Filters the list of single blocks allowed for custom post type posts.
		 *
		 * @param array $choices
		 */
		$choices = (array) apply_filters( 'wpex_cpt_single_blocks_choices', $choices );

		return $choices;
	}

	/**
	 * Returns custom post type post blocks to display.
	 */
	public static function get( $post_type = '' ) {
		if ( ! $post_type ) {
			$post_type = get_post_type();
		}

		$default_blocks = array(
			'media',
			'title',
			'meta',
			'post-series',
			'content',
			'page-links',
			'share',
			'comments',
		);

		$blocks = get_theme_mod( $post_type . '_single_blocks', $default_blocks );

		if ( WPEX_PTU_ACTIVE ) {
			$ptu_check = wpex_get_ptu_type_mod( $post_type, 'single_blocks' );
			if ( $ptu_check ) {
				$blocks = $ptu_check;
			}
		}

		if ( is_string( $blocks ) ) {
			$blocks = $blocks ? explode( ',', $blocks ) : array();
		}

		if ( $blocks ) {

			/*
			 * Make sure only blocks registered to be selected in the customizer can render when
			 * pulled from the theme_mod.
			 */
			$blocks = array_intersect( $blocks, array_keys( self::choices() ) );
		}

		// Set keys equal to vars for easier unsetting via hooks.
		$blocks = array_combine( $blocks, $blocks );

		if ( defined( 'TYPES_VERSION' ) && $blocks ) {
			foreach ( $blocks as $block ) {
				if ( ! get_theme_mod( 'cpt_single_block_' . $block . '_enabled', true ) ) {
					unset( $blocks[$block] );
				}
			}
		}

		/**
		 * Filters the custom post type single post blocks.
		 *
		 * @param array $blocks
		 */
		$blocks = apply_filters( "wpex_{$post_type}_single_blocks", $blocks, $post_type );

		/**
		 * Filters the custom post type single post blocks.
		 *
		 * @param array $blocks
		 */
		$blocks = (array) apply_filters( 'wpex_single_blocks', $blocks, $post_type );

		return $blocks;
	}

	/**
	 * Render CPT post blocks.
	 */
	public static function render() {
		$blocks = self::get();

		if ( empty( $blocks ) || ! is_array( $blocks ) ) {
			return;
		}

		foreach ( $blocks as $block ) {
			if ( 'media' === $block && wpex_has_custom_post_media_position() ) {
				continue; // Media not needed for this position.
			}
			if ( 'the_content' !== $block && is_callable( $block ) ) {
				call_user_func( $block );
			} else {
				get_template_part( 'partials/cpt/cpt-single-' . $block, get_post_type() );
			}

		}
	}

	/**
	 * Echo class attribute for the single custom post type blocks wrapper element.
	 */
	public static function wrapper_class() {
		$class = array(
			'wpex-first-mt-0',
			'wpex-clr',
		);

		/**
		 * Filters the custom post type single blocks element class.
		 *
		 * @param array $class.
		 */
		$class = (array) apply_filters( 'wpex_cpt_single_blocks_class', $class );

		if ( $class ) {
			echo 'class="' . esc_attr( implode( ' ', $class ) ) . '"';
		}
	}

}